package k.console.table

import k.common.*
import java.security.InvalidParameterException

const val dateFormat = "yyyy-MM-dd hh:mm:ss"

enum class ColumnType(val align : Align, val minWidth : Int, private val fmt : (value : Any, fmt : CellFormat, width : Int) -> String, val noWrap : Boolean = false)
{
    Text(Align.Left, 3, { v, _, _ -> v.str }),
    ID(Align.Left, 3, { v, _, _ -> v.str }),
    Date(Align.Center, dateFormat.length,
         { v, _, _ ->
             val date = when (v)
             {
                 is Long           -> java.util.Date(v)
                 is java.util.Date -> v
                 else              -> throw java.security.InvalidParameterException()
             }

             java.text.SimpleDateFormat(dateFormat).format(date)
         }),
    Progress(Align.Left, 10,
             { v, f, w ->
                 when (v)
                 {
                     is String -> v
                     is Double ->
                     {
                         f.accent = AnsiColor.Undefined
                         f.textColor = AnsiColor.Blue
                         f.color = AnsiColor.White

                         val value = (v * w).int

                         ' '.pad(value).conFormat(backColor = AnsiColor.Blue) + ' '.pad(w - value).conFormat(backColor = AnsiColor.White)
                     }

                     else      -> throw InvalidParameterException()
                 }
             }, noWrap = true),
    DurationStr(Align.Right, 6,
                { v, _, _ ->
                    val duration = when (v)
                    {
                        is Duration -> v
                        is Long     -> Duration(v)
                        is String   -> Duration(v)

                        else        -> throw InvalidParameterException()
                    }

                    duration.value.negative.choose("~", duration.str)
                }),
    Percent(Align.Right, 6, { v, _, _ -> (v as Double).percentStr }),
    Count(Align.Right, 5,
          { v, _, _ ->
              when (v)
              {
                  is Long   -> v.amount
                  is Int    -> v.amount
                  is String -> v.long.amount

                  else      -> throw InvalidParameterException()
              }
          }),
    Number(Align.Right, 5, { v, _, _ -> v.str });

    operator fun invoke(value : Any, fmt : CellFormat, width : Int) =
        fmt(value, fmt, width)
}