package k.ofl

import k.common.*
import k.ofl.parse.OflParser
import k.stream.text
import java.io.File

class OFLError(message : String) : Error(message)

/**
 * Object flow language.
 *
 * Usable for application configuration and simple scripting
 */

class OFL() : OflObject()
{
    companion object
    {
        val EMPTY = OFL()
    }

    /**
     * Text to parsed object
     */
    constructor(str : String) : this()
    {
        parse(str)
    }

    private var initLog : String = ""

    fun onApplicationStart(args : Array<String>, appTitle : String) : OFL
    {
        stage("Parse command line")
        {
            var index = 0

            args
                .map { it.trim() }
                .filter { it.isNotBlank() }
                .forEach {
                    when (it[0])
                    {
                        '-', '/' ->
                        {
                            val parts = it.drop(1).split('=')

                            when (parts.size)
                            {
                                1 -> set(parts[0], true)
                                2 -> set(parts[0], parts[1].trim('"', '\''))
                            }
                        }

                        else     ->
                        {
                            set("$index", it.trim('"', '\''))
                            index++
                        }
                    }
                }
        }

        loadCfgResource("application-const.ofl")

        msg(appTitle, MsgType.BlueText)
        msg(" ${appConfig["ImplementationVersion"]}".n.n)
        msg(initLog, MsgType.Debug)

        return this
    }

    /**
     * Default initialization of application configuration.
     *
     * Priority:
     *
     * 1. Commandline parameters
     * 2. JRE.System parameters
     * 3. Environment variables
     * 4. Application.ofl file in working directory
     * 5. Application.ofl file in resources
     */
    fun loadConfiguration() : OFL
    {
        val cfgFile = "application.ofl"

        arrayOf(cfgFile, "application-lib.ofl", "application-test.ofl").forEach { fileName ->
            loadCfgResource(fileName)
        }

        if (File(cfgFile).exists())
        {
            this += OFL(File(cfgFile).readText())

            initLog += "External configuration loaded".n
        }

        this += System.getenv()

        initLog += "Environment configuration loaded".n

        System.getProperties().forEach { set(it.key.str, it.value.str) }

        initLog += "System configuration loaded".n

        loadCfgResource("application-const.ofl")

        this["ImplementationVersion"] = OFL::class.java.`package`.implementationVersion
                                        ?: "[debug version]"

        return this
    }

    private fun loadCfgResource(name : String)
    {
        if (isResourceExists(name))
        {
            this += OFL(resource(name).text)

            initLog += "Internal configuration loaded".n
        }
    }

    override val level : Int
        get() = 0

    private fun parse(str : String)
    {
        clear()
        OflParser(str, this)
    }

    override val str
        get() = childStr

    override fun fmt(offset : String, alignPos : Int) : String =
        childFmt(offset).trim()

    override val json
        get() = "{\n$childJson}"
}