package k.serializing

import k.common.*
import k.ofl.*
import java.lang.reflect.Field
import java.util.*

abstract class Serializer<S : Any> {
    abstract fun isNativeValue(value : Any) : Boolean
    abstract fun setValue(obj : S, name : String, value : String)
    abstract fun setNativeValue(name : String, obj : S, value : Any)
    abstract fun getObj(root : S, path : String) : S
    abstract fun getList(root : S, path : String) : S

    protected var level = -1

    fun <T : Any> serialize(target : S, value : T, forHuman : Boolean, keyField : Field?) {
        level++

        var type = value.javaClass

        do {
            type.declaredFields
                .filter { (it != keyField && it.annotation<Hide>().isNull) }
                .map { field ->
                    val fieldName = field.annotation<Name>()?.value default field.name
                    val fieldType = field.type

                    if (fieldType != type && !fieldType.name.endsWith("\$Companion"))
                        replaceError("Failed to marshall: $type::$fieldName ($fieldType)") {
                            field.isAccessible = true
                            val fieldValue = field.get(value)

                            packValue(target, fieldName, fieldType, fieldValue, forHuman, field.annotation<Optional>()?.value)
                        }
                }

            type = type.superclass.cast()
        } while (type.classLoader.isNotNull)

        level--
    }

    private fun <T : Any> packValue(obj : S, fieldName : String, fieldType : Class<*>, fieldValue : T, forHuman : Boolean, optional : String?) {
        fun put(value : String) =
            setValue(obj, fieldName, value)

        if (fieldValue.isNotNull && isNativeValue(fieldValue))
            setNativeValue(fieldName, obj, fieldValue)
        else
            when (fieldType) {
                OflObject::class.java,
                OFL::class.java      -> {
                    val ofl = fieldValue.cast<OflObject>()

                    if (ofl.items.isNotEmpty() || optional.isNull) {
                        put(if (forHuman)
                                ofl.fmt
                            else
                                ofl.str)
                    }
                }

                Duration::class.java -> {
                    if (forHuman)
                        put(fieldValue.str)
                    else
                        put((fieldValue as Duration).value.str)
                }

                Date::class.java     -> {
                    if (forHuman)
                        put(fieldValue.str)
                    else
                        put((fieldValue as Date).time.str)
                }

                Map::class.java      -> {
                    val map = fieldValue.cast<Map<Any, Any>>()

                    if (map.isNotEmpty() || optional.isNull) {
                        val fieldObj = getObj(obj, fieldName)

                        level++

                        map.entries
                            .sortedBy { it.key.toString() }
                            .forEach {
                                packValue(fieldObj, it.key.toString(), it.value.javaClass, it.value, forHuman, null)
                            }

                        level--
                    }
                }

                LinkedList::class.java,
                ArrayList::class.java,
                List::class.java     -> {
                    val list = fieldValue.cast<List<Any>>()

                    if (list.isNotEmpty() || optional.isNull) {
                        val firstItem = list.firstOrNull()

                        if (this is NamedItems && firstItem?.javaClass?.classLoader != null) {
                            val fieldObj = getObj(obj, fieldName)
                            val keyField = firstItem.javaClass.declaredFields.first()
                            keyField.isAccessible = true

                            level++

                            list.forEach {
                                serialize(getObj(fieldObj, keyField.get(it).toString()), it, forHuman, keyField)
                            }

                            level--
                        }
                        else {
                            val fieldObj = getList(obj, fieldName)

                            level++

                            list.forEach {
                                packValue(fieldObj, "", it.javaClass, it, forHuman, null)
                            }

                            level--
                        }
                    }
                }

                else                 ->
                    if (fieldType.classLoader.isNotNull && !fieldType.isEnum)
                        serialize(getObj(obj, fieldName), fieldValue, forHuman, null)
                    else {
                        val valueStr = fieldValue.str

                        if (optional == null || optional != valueStr)
                            put(valueStr)
                    }
            }
    }
}