package k.serializing.json

import k.common.orThrow

const val openObject = '{'
const val closeObject = '}'
const val closeArray = ']'
const val openArray = '['
const val valueDivider = ':'
const val divider = ','
const val space = ' '
const val tab = '\t'
const val quote = '"'
const val mask = '\\'

const val closeValue = "$closeObject$closeArray$divider$valueDivider$space$tab$quote"

internal class ResistantJson(val text : String) {
    private var cursor = 0
    private var start = 0

    val root = parse()

    private fun parse() : JsonContainer {
        val items = mutableListOf<JsonContainer>()
        val properties = mutableListOf<JsonNode>()

        while (cursor < text.length) {
            when (val char = text[cursor]) {
                openArray           -> items += parseArray("")
                openObject          -> items += parseObject("")

                space, tab, divider -> cursor++
                else                -> if (char.isWhitespace())
                    cursor++
                else
                    properties += parseNode()
            }
        }

        (items.isEmpty() || properties.isEmpty()) orThrow "Objects and properties in root detected"

        return if (properties.isEmpty())
            when (items.size) {
                0    -> JsonObject("", listOf(JsonProperty("Content", text)))
                1    -> items.first()
                else -> JsonArray("", items)
            }
        else
            JsonObject("", properties)
    }

    private fun parseArray(name : String) : JsonArray {
        cursor++

        val items = mutableListOf<JsonNode>()

        while (cursor < text.length) {
            when (val char = text[cursor]) {
                closeArray, closeObject -> {
                    cursor++
                    break
                }

                openObject              -> items += parseObject("")
                openArray               -> items += parseArray("")

                space, tab, divider     -> cursor++
                quote                   -> items += JsonNode(parseString())

                else                    -> if (char.isWhitespace())
                    cursor++
                else
                    items += JsonNode(parseValue())
            }
        }

        return JsonArray(name, items)
    }

    private fun parseObject(name : String) : JsonObject {
        cursor++

        val children = mutableListOf<JsonNode>()

        while (cursor < text.length) {
            when (val char = text[cursor]) {
                closeObject         -> {
                    cursor++
                    break
                }

                space, tab, divider -> cursor++

                else                -> if (char.isWhitespace())
                    cursor++
                else
                    children += parseNode()
            }
        }

        return JsonObject(name, children)
    }

    private fun parseNode() : JsonNode {
        val name = if (text[cursor] == '"')
            parseString()
        else
            parseValue()

        while (cursor < text.length) {
            when (val char = text[cursor]) {
                openObject               -> return parseObject(name)
                openArray                -> return parseArray(name)
                space, tab, valueDivider -> cursor++
                quote                    -> return JsonProperty(name, parseString())

                else                     -> if (char.isWhitespace())
                    cursor++
                else
                    return JsonProperty(name, parseValue())
            }
        }

        return JsonProperty(name, "")
    }

    private fun parseString() : String {
        start = ++cursor

        while (cursor < text.length) {
            if (text[cursor] == quote && text[cursor - 1] != mask)
                break
            else
                cursor++
        }

        return text.substring(start, cursor++)
    }

    private fun parseValue() : String {
        start = cursor

        while (cursor < text.length) {
            val char = text[cursor]

            if (char in closeValue || char.isWhitespace())
                break

            cursor++
        }

        return text.substring(start, cursor)
    }
}