package k.stream

import k.common.*
import java.nio.ByteBuffer

const val chunkSize = KB.toInt()

class MemoryStorage(size: Long = 0) : StreamStorage
{
    override var size: Long = size
        set(value)
        {
            val chunkCount = chunkNumber(value) + 1

            while (chunks.count() < chunkCount)
                chunks += ByteBuffer.allocate(chunkSize)

            while (chunks.count() > chunkCount)
                chunks.removeLast()

            field = value
        }

    override fun read(buf: ByteBuffer, position: Long)
    {
        proceedChunks(position, buf) { chunk, bufPos, chunkPos, chunkSize ->
            buf.put(bufPos, chunk, chunkPos, chunkSize)
        }
    }

    override fun write(buf: ByteBuffer, position: Long)
    {
        proceedChunks(position, buf) { chunk, bufPos, chunkPos, chunkSize ->
            chunk.put(chunkPos, buf, bufPos, chunkSize)
        }
    }

    override fun flush()
    {
        // Nothing
    }

    override fun close()
    {
        // Nothing
    }

    override val connected
        get() = true

    private val chunks = mutableListOf<ByteBuffer>()

    private fun chunkNumber(position: Long) =
        (position / chunkSize).int

    private fun chunkPosition(number: Int) =
        number.toLong() * chunkSize

    private fun proceedChunks(position : Long, buf : ByteBuffer, chunkCode : (chunk : ByteBuffer, bufPos : Int, chunkPos : Int, chunkSize : Int) -> Unit)
    {
        var curPosition = position
        var remainReadSize = buf.limit() - buf.position()
        var chunkNumber = chunkNumber(curPosition)
        var startPos = (curPosition - chunkPosition(chunkNumber)).int

        do
        {
            val curChunk = chunks[chunkNumber]

            chunkNumber++

            val piece = Integer.min(remainReadSize, (chunkPosition(chunkNumber) - curPosition).int)

            curChunk.position(startPos)

            chunkCode(curChunk, buf.limit() - remainReadSize, startPos, piece)

            startPos = 0
            curPosition += piece
            remainReadSize -= piece
        } while (remainReadSize > 0)
    }
}