package k.common

import k.parallels.Sleepy
import java.lang.Thread.startVirtualThread
import java.util.concurrent.atomic.*

fun backgroundTask(desc : String, parentLog : LogItem, period : Duration = 1.sec, debounce : Duration = 200.ms, code : (LogItem) -> Unit) =
    BackgroundTask(desc, parentLog, period, debounce, code)

class BackgroundTask(desc : String, parentLog : LogItem, period : Duration, debounce : Duration, code : (LogItem) -> Unit) {
    val log = parentLog.post("Root of $desc")
    private val generation = AtomicInteger(0)
    private val active = AtomicBoolean(true)
    private val sleepy = Sleepy()

    private val thread = startVirtualThread {
        while (active.get()) {
            val startGeneration = generation.get()

            log.rootStage(desc) {
                code(it)
            }

            mute {
                sleepy.wakeUp()

                val timer = Timer()

                do {
                    deBounce(debounce)
                } while (active.get() && startGeneration == generation.get() && timer.time < period)
            }
        }
    }

    private fun deBounce(period : Duration) {
        do {
            val startGeneration = generation.get()

            period.sleep()
        } while (active.get() && startGeneration != generation.get())
    }

    fun activate(wait : Boolean = false) {
        generation.incrementAndGet()

        if (wait)
            waitFor()
    }

    fun waitFor() =
        sleepy.waitFor()

    fun stop() {
        active.set(false)
        thread.interrupt()
        thread.join()
    }
}