package k.common

import k.parallels.Sync
import k.serializing.*
import k.serializing.Optional
import java.util.*

const val maxStatusLen = 10
const val defaultLogCapacity = 100.toShort()

enum class LogStatus(val canContinue : Boolean,
                     val isEnough : Boolean,
                     val isSuccess : Boolean,
                     val type : MsgType) {
    Stage(true, true, true, MsgType.Text),
    Info(true, true, true, MsgType.BlueText),
    Success(true, true, true, MsgType.Ok),
    Warning(true, true, false, MsgType.Warning),
    Error(true, false, false, MsgType.Error),
    FatalError(false, false, false, MsgType.Error);
}

class LogContext {
    val sync = Sync()
}

class Log : LogItem(null,
                    LogContext(),
                    "[Root]",
                    "",
                    LogStatus.Success,
                    defaultLogCapacity,
                    0) {
    companion object {
        val EMPTY = Log()
    }
}

const val offset = 2

open class LogItem(message : String, details : String, status : LogStatus, time : Long, duration : Long, items : LinkedList<LogItem>) {
    val message = message

    @Optional
    val details = details

    var status = status
        private set

    var time = time
        private set

    var duration = duration
        private set

    val items = items

    @Hide
    private var parent : LogItem? = null

    @Hide
    private lateinit var context : LogContext

    @Hide
    var capacity : Short = defaultLogCapacity

    constructor(parent : LogItem?,
                context : LogContext,
                message : String,
                details : String,
                status : LogStatus,
                capacity : Short,
                duration : Long) : this(message, details, status, System.currentTimeMillis(), duration, LinkedList<LogItem>()) {
        this.parent = parent
        this.context = context
        this.capacity = capacity
    }

    private fun propagateStatus(status : LogStatus) {
        if (this.status < status) {
            this.status = status

            parent?.propagateStatus(status)
        }
    }

    fun updateStatus() : LogStatus {
        status = context.sync
            .read { items.maxOfOrNull { it.updateStatus() } }
            ?: status

        return status
    }

    val level : Int
        get() = 1 + (parent?.level ?: -1)

    fun post(message : String, status : LogStatus = LogStatus.Info, details : String = "") : LogItem {
        val item = LogItem(this, context, message, details, status, capacity, 0)

        msg(item.toString().n, status.type)

        context.sync {
            propagateStatus(status)

            items += item

            if (items.size > capacity)
                items.removeFirst()
        }

        return item
    }

    operator fun plusAssign(value : LogItem) =
        context.sync {
            propagateStatus(value.status)

            items += value.items

            while (items.size > capacity)
                items.removeFirst()
        }

    fun catch(vararg code : () -> Unit) =
        code.forEach {
            try {
                it()
            }
            catch (e : Throwable) {
                post(e.str, LogStatus.Error)

                return@catch
            }
        }

    infix fun catch(code : () -> Unit) =
        try {
            code()
        }
        catch (e : Throwable) {
            post(e.str, LogStatus.Error, e.fmtStackTrace)
        }

    fun <T> catch(code : () -> T, default : T) =
        try {
            code()
        }
        catch (e : Throwable) {
            post(e.str, LogStatus.Error, e.fmtStackTrace)
            default
        }

    fun <T> rootStage(desc : String, failStatus : LogStatus = LogStatus.FatalError, code : (LogItem) -> T) =
        rootStageInt(desc, { null }, failStatus, code)

    fun <T> rootStage(desc : String, default : (e : Throwable) -> T, failStatus : LogStatus = LogStatus.FatalError, code : (LogItem) -> T) =
        rootStageInt(desc, default, failStatus, code)!!

    private inline fun <T> rootStageInt(desc : String,
                                        default : (e : Throwable) -> T?,
                                        failStatus : LogStatus = LogStatus.FatalError,
                                        code : (LogItem) -> T) : T? {
        val log = post(desc, LogStatus.Stage)
        log.status = LogStatus.Success

        val start = System.nanoTime()

        return try {
            val res = code(log)

            log.duration = System.nanoTime() - start

            res
        }
        catch (e : Throwable) {
            log.duration = System.nanoTime() - start

            log.post(e.str, failStatus, e.fmtStackTrace)

            default(e)
        }
    }

    fun <T> stage(desc : String, failStatus : LogStatus = LogStatus.FatalError, code : (LogItem) -> T) =
        if (status.canContinue)
            rootStage(desc, failStatus, code)
        else
            null

    fun <T> stage(desc : String, default : (e : Throwable) -> T, failStatus : LogStatus = LogStatus.FatalError, code : (LogItem) -> T) =
        if (status.canContinue)
            rootStage(desc, default, failStatus, code)
        else
            null

    val firstError : LogItem?
        get() = context.sync
            .read {
                items.firstNotNullOfOrNull { if (it.status.isEnough) null else it.firstError }
            }
            ?: if (status.isEnough) null else this

    private val prefixStr
        get() = "${Date(time).asDateTimeStr} ${' '.pad(level * offset)}[$status]${' '.pad(maxStatusLen - status.str.length)}"

    private val prefixSpace
        get() = ' '.pad(prefixStr.length + 1)

    override fun toString() =
        "$prefixStr $message\n${(prefixSpace and details.replace("\n", "\n$prefixSpace")) merge "\n"}${items.joinToString("\n") { "$it" }}".trimEnd()

    val json : String
        get() = context.sync
            .read {
                val childs = items.isNotEmpty() accept { ""","items":[${items.joinToString(",") { it.json }}]""" }

                return """{"msg":"$message"${details.isNotEmpty() accept ""","details":"$details""""},"time":$time,"status":"$status","duration":$duration$childs}"""
            }
}