package k.common

import k.stream.text
import java.io.File
import java.lang.Thread.sleep
import kotlin.concurrent.thread

/**
 * The outer quotes of each parameter of commandLine are removed before starting the process
 */
class ProcessRunner(commandLine : String,
                    environmentVariables : Map<String, String> = mapOf(),
                    workDir : String = "") {
    private val process = ProcessBuilder(commandLine.cmdParams)
        .redirectErrorStream(true)
        .redirectOutput(ProcessBuilder.Redirect.PIPE)
        .also {
            it.environment() += environmentVariables

            if (workDir.isNotBlank())
                it.directory(File(workDir))
        }
        .start()

    val handle : ProcessHandle
        get() = process.toHandle()

    var output : String = ""
        get() {
            mute {
                if (process.inputStream.available() > 0)
                    field += process.inputStream.text
            }

            return field
        }
        private set

    val exitCode
        get() = process.exitValue()

    val running
        get() = process.isAlive

    fun wait(timeout : Duration = Duration.INFINITE) : Boolean {
        val timer = Timer()

        if (timeout != Duration.INFINITE)
            thread {
                while (running && timer.time < timeout)
                    sleep(100)

                process.children()
                    .forEach { it.destroy() }

                process.destroy()
            }

        while (running)
            output

        return !running && (exitCode == 0)
    }
}

fun cmdLine(cmd : String, workDir : File? = null, timeout : Duration = 1.min, silent : Boolean = false) : String =
    ProcessRunner(cmd, workDir = workDir?.absolutePath default "").let { process ->
        return if (process.wait(timeout) || silent)
            process.output.trim()
        else
            error("Failed to execute command line: $cmd\n\nWith output:\n\n${process.output.trim()}")
    }

val String.cmdParams : List<String>
    get() {
        val params = mutableListOf<String>()
        val builder = StringBuilder()
        var quoteChar : Char? = null

        fun onParamFinished() {
            if (builder.isNotEmpty()) {
                params.add(builder.toString())
                builder.clear()
            }
        }

        var prevChar = ' '

        forEach { char ->
            if (char.isWhitespace()) {
                if (quoteChar.isNull)
                    onParamFinished()
                else
                    builder.append(char)
            }
            else if (char in "\"'" && prevChar != '\\') {
                if (char == quoteChar) {
                    onParamFinished()

                    quoteChar = null
                }
                else if (quoteChar.isNull)
                    quoteChar = char
                else
                    builder.append(char)
            }
            else if (!char.isWhitespace())
                builder.append(char)

            prevChar = char
        }

        onParamFinished()

        return params
    }