package k.console.table

import k.common.*
import k.console.Console

private val linesLimit by lazy { appConfig["Console.Table.Cell.linesLimit", "4"].int }
private val tableWidth by lazy { appConfig["Console.Table.Width", "105"].int }
private const val LEVEL_OFFSET = 2

class Table<R>(private vararg val columns : Column<R>,
               val padding : Int = 0,
               val width : Int = tableWidth,
               val limit : Int = linesLimit,
               var childGetter : ((row : R) -> List<R>)? = null) {
    private class CellValue(val lines : List<String>,
                            val fmt : CellFormat,
                            val levelOffset : Int = 0)

    private class ColFit<R>(val col : Column<R>,
                            val rows : List<CellValue>,
                            var width : Int,
                            var gapStr : String = "",
                            val fmtStr : String = "")

    private lateinit var fits : List<ColFit<R>>

    private class TreeRow<R>(val row : R, val level : Int)

    private fun expand(it : R, level : Int) : List<TreeRow<R>> =
        listOf(TreeRow(it, level)) +
                (childGetter?.let { getter -> getter(it) }
                    ?.map { expand(it, level + 1) }
                    ?.flatten()
                    .orEmpty())

    /**
     * Print table
     */
    fun print(rows : List<R>, showHeader : Boolean = true) : Int {
        val expandedRows = rows.map { expand(it, 0) }.flatten()

        calculateFits(expandedRows)

        if (showHeader)
            showHeader()

        var odd = false

        repeat(expandedRows.size) {
            if (expandedRows[it].level == 0 && childGetter != null) {
                if (it.positive)
                    Console.newLine()

                odd = true
            }

            showRow(odd, it, expandedRows[it].row)

            odd = !odd

            Console.newLine()
        }

        return expandedRows.size
    }

    private fun getValue(row : R, col : Column<R>, width : Int, levelOffset : Int) : CellValue {
        val fmt = CellFormat()

        return CellValue((col.type(col.getter(row, fmt), fmt, width) - "\r")
                             .split("\n")
                             .flatMap { if (it.isEmpty() || col.type.noWrap) listOf(it) else it.chunked(width) },
                         fmt,
                         levelOffset)
    }

    private fun calculateFits(rows : List<TreeRow<R>>) {
        var lineLimit = width - columns.count() * 3

        do {
            fits = columns
                .filter { it.visible }
                .mapIndexed { idx, col ->
                    col to rows.map { treeRow ->
                        val cellOffset = (idx == 0).choose(treeRow.level * LEVEL_OFFSET, 0)

                        getValue(treeRow.row, col, lineLimit - cellOffset, cellOffset)
                    }
                }
                .map {
                    ColFit(it.first,
                           it.second,
                           listOf((if (it.second.isEmpty()) 0 else it.second.flatMap { value -> value.lines.map { line -> value.levelOffset + line.length } }.max()),
                                  it.first.header.length,
                                  it.first.minWidth,
                                  it.first.type.minWidth).max()
                          )
                }

            val tablePadding = 1 + 1
            val cellLeftPadding = 1
            val gapCount = (fits.count() - 1)
            val usedSpace = fits.sumOf { it.width } + tablePadding + gapCount * cellLeftPadding
            val freeSpace = width - usedSpace
            val gap = freeSpace / gapCount

            lineLimit--

            if (gap > 0) {
                fits.first().gapStr = " "

                if (fits.count() == 1)
                    fits.first().width = width
                else
                    for (i in 1..gapCount)
                        fits[i].gapStr = ' '.pad(gap + 1)

                val lostGap = freeSpace - gap * gapCount

                fits.maxBy { it.width }.width += lostGap

                break
            }
        } while (rows.isNotEmpty())
    }

    private fun showHeader() {
        print(' '.pad(padding))
        print("".conFormat(AnsiColor.Black, AnsiColor.Gray))

        fits.forEach { print(it.gapStr + it.col.type.align(it.col.header, it.width)) }

        print(" $resetConFormat".n)
    }

    fun showRow(odd : Boolean, rowIdx : Int, row : R) {
        var line = 0
        var hasNextLine = true

        while (hasNextLine && line <= limit) {
            print(resetConFormat)
            print(' '.pad(padding))

            hasNextLine = false

            fits.forEach {
                val cell = if (it.rows.isEmpty())
                    getValue(row, it.col, it.width, 0)
                else
                    it.rows[rowIdx]

                print(' '.pad(cell.levelOffset))

                print(odd.choose("".conFormat(AnsiColor.Black, AnsiColor.DarkGray),
                                 "".conFormat(AnsiColor.White, AnsiColor.Default)))
                print(it.gapStr)

                print(odd.choose("".conFormat(cell.fmt.textColor or AnsiColor.Black, cell.fmt.color or cell.fmt.accent or AnsiColor.DarkGray),
                                 "".conFormat(cell.fmt.textColor or cell.fmt.accent or AnsiColor.White, cell.fmt.color or AnsiColor.Default)))

                if (line < cell.lines.size) {
                    val contentWith = it.width - cell.levelOffset

                    val valueStr = (line == limit).choose(Align.Center(AnsiColor.Orange("..."), contentWith),
                                                          it.col.type.align(cell.lines[line], contentWith))

                    print(valueStr)

                    if (cell.lines.size > line + 1)
                        hasNextLine = it.rows.isNotEmpty()
                }
                else
                    print(' '.pad(it.width))
            }

            line++

            print(" $resetConFormat")

            if (hasNextLine)
                Console.newLine()
        }
    }
}