package k.ofl

import k.common.*

const val contentOffset = 2

open class OflObject : OflItem() {
    override val key
        get() = name.ifEmpty { children.firstOrNull()?.value.orEmpty() }

    open val startChar = '{'
    open val finishChar = '}'
    protected val children = mutableListOf<OflItem>()

    val asObjects
        get() = children.filterIsInstance<OflObject>()

    val items
        get() = children.toList()

    fun clear() =
        children.clear()

    internal fun doClone(item : OflObject) : OflObject {
        item.name = name

        forEach { item.add(it.clone) }

        return item
    }

    fun isFeatureOn(name : String) =
        !(find(name, OflRecursive.Up)?.value?.same("false")
            ?: true)

    override val clone
        get() : OflItem = doClone(OflObject())

    val childJson
        get() = children.joinToString(",\n") { it.json }

    override val json
        get() = "${
            if (parent is OflArray) ""
            else if (super.name.isEmpty()) "" else "${super.json}:"
        }$startChar\n$childJson$finishChar"

    val count
        get() = children.count()

    val isEmpty
        get() = count == 0

    val last
        get() = children.last()

    val childStr
        get() = children.joinToString("\n") { it.str }

    fun childFmt(offset : String) : String {
        var maxLen = (children.maxOfOrNull { it.title.length } ?: 0)

        if (maxLen > 0)
            maxLen++

        return children.filter { it !is OflObject }.joinToString("\n") { it.fmt(offset, maxLen) } +
                ("\n\n" and children.filterIsInstance<OflObject>().joinToString("\n\n") { it.fmt(offset, maxLen) })
    }

    override val values
        get() = children.map { it.value }.toTypedArray()

    override val str
        get() = "${super.str}$startChar$childStr$finishChar"

    override fun fmt(offset : String, alignPos : Int) : String {
        if (name.isEmpty() && count == 0)
            return ""

        val childStr = childFmt(offset + ' '.pad(alignPos + contentOffset)).trim('\n')

        val firstLine = if (name.isEmpty())
            "$offset$startChar"
        else
            "$offset$title${' '.pad(alignPos - title.length)}$startChar"

        return firstLine + if (count == 0)
            finishChar
        else if ('\n' in childStr)
            "\n$childStr\n$offset${' '.pad(alignPos)}$finishChar"
        else
            " ${childStr.trim()} $finishChar"
    }

    enum class OflRecursive() {
        Up,
        None,
        Down
    }

    fun getOneOf(vararg names : String) : String {
        names.forEach {
            find(it)?.let { res ->
                return res.value
            }
        }

        throw NotFoundError("${path.dot}${names.first()} not found")
    }

    open fun find(name : String) =
        children.find { it.name same name }

    open fun find(name : String, recursive : OflRecursive) : OflItem? {
        val parent = this.parent

        children.forEach { child ->
            if (child.name same name)
                return child

            if (child is OflObject && recursive == OflRecursive.Down)
                child.find(name, recursive)?.let { return it }
        }

        return if (recursive == OflRecursive.Up && parent is OflObject)
            parent.find(name, recursive)
        else
            null
    }

    fun forceObj(name : String) : OflObject {
        var item = find(name)

        if (item != null)
            if (item is OflObject)
                return item
            else
                remove(item)

        item = add(OflObject()) as OflObject
        item.name = name

        return item
    }

    fun forceArr(name : String) =
        (find(name)
            ?: add(OflArray()).also {
                it.name = name
            }) as OflArray

    fun forEachObj(name : String = "*", recursive : Boolean = false, code : (it : OflObject) -> Unit) =
        forEach(name, recursive) {
            if (it is OflObject)
                code(it)
        }

    fun forEachProp(name : String = "*", recursive : Boolean = false, code : (it : OflProperty) -> Unit) =
        forEach(name, recursive) {
            if (it is OflProperty)
                code(it)
        }

    fun forEach(name : String = "*", recursive : Boolean = false, code : (it : OflItem) -> Unit) {
        children.forEach {
            if (name == "*" || it.name.same(name))
                code(it)

            if (recursive && it is OflObject)
                it.forEach(name, true, code)
        }
    }

    operator fun invoke(index : Int) =
        children[index]

    operator fun get(index : Int) =
        children[index] as OflObject

    operator fun invoke(name : String, recursive : OflRecursive = OflRecursive.None) =
        find(name, recursive) mustBeFound "${path.dot}$name"

    operator fun get(name : String, default : String? = null) =
        getOrNull(name)?.value default { default mustBeFound "${path.dot}$name" }

    fun getOrNull(name : String, recursive : OflRecursive = OflRecursive.None) : OflItem? {
        find(name)
            ?.let { return it }

        val parts = splitPath(name)
        var item = this

        return item
            .find(parts.first(), recursive)
            ?.let { root ->
                if (parts.size == 1)
                    return root

                if (root is OflObject)
                    item = root
                else
                    return null

                for (i in 1..parts.size - 2) {
                    val curItem = item.find(parts[i])

                    if (curItem is OflObject)
                        item = curItem
                    else
                        return null
                }

                item.find(parts.last())
            }
    }

    /*private fun obj(name : String) : OflObject {
        val parts = splitPath(name)
        var item = this

        parts.forEach {
            item = (item.find(it) mustBeFound "${path.dot}$name") as OflObject
        }

        return item
    }*/

    operator fun contains(name : String) =
        (find(name, OflRecursive.None) != null)

    operator fun set(name : String, value : String) {
        val parts = splitPath(name)
        var child : OflObject = this

        for (i in 0..parts.size - 2)
            child = child.forceObj(parts[i])

        child.remove(child.find(parts.last()))

        val item = OflProperty()
        item.name = parts.last()
        item.value = value

        child.add(item)
    }

    fun remove(item : OflItem?) {
        if (item != null)
            children.remove(item)
    }

    open fun add(name : String, value : String) : OflProperty {
        remove(find(name))

        val property = OflProperty(name, value)

        property.parent = this

        children.add(property)

        return property
    }

    open fun add(item : OflItem) : OflItem {
        remove(find(item.name))

        val clone = if (item.parent == null)
            item
        else
            item.clone

        clone.parent = this

        children.add(clone)

        return item
    }

    open fun put(item : OflItem) {
        find(item.name).isNull orThrow {
            OFLError("[${path default "Root"}.${item.name}] Duplicate property")
        }

        item.parent = this

        children.add(item)
    }

    operator fun set(name : String, value : OflItem) {
        val item = value.clone
        item.name = name

        add(item)
    }

    operator fun <T> set(name : String, value : T) {
        this[name] = value.toString()
    }

    fun <T> setDefault(name : String, value : T) {
        val n = name.low

        if (n !in this)
            this[n] = value
    }

    operator fun plus(value : OflObject) =
        OFL().also { res ->
            this.items.forEach { res.add(it) }
            value.items.forEach { res.add(it) }
        }

    operator fun plusAssign(vars : OflObject) =
        vars.forEach { add(it) }

    operator fun plusAssign(vars : Map<String, String>) =
        vars.forEach {
            this[it.key] = it.value
        }

    private fun splitPath(path : String) =
        path.split('.')

    override var value = ""
        get() = childStr
}