package k.docker

import k.common.*
import k.docker.models.*
import k.serializing.*
import org.http4k.core.*
import org.http4k.filter.ClientFilters
import java.io.File
import java.lang.Integer.min
import java.time.OffsetDateTime
import java.time.format.DateTimeFormatter
import java.util.*

enum class ContainerRunMode(val cmd : String) {
    Default(""),
    AutoRestart("--restart=always"),
    AutoRemove("--rm");
}

const val divider = "~@~"

object Docker {
    val String.id
        get() = this
            .lines()
            .map { it.trim() }
            .lastOrNull { it.isNotEmpty() } ?: ""

    fun createContainer(service : Service,
                        labels : ParamList? = null,
                        command : String = "",
                        mode : ContainerRunMode = ContainerRunMode.Default,
                        run : Boolean = false) =
        cmdLine("docker " + (if (run) "run " else "create ") +
                        " ${mode.cmd} " +
                        (command accept " --entrypoint sh ") +
                        " --name ${service.name} " +
                        service.mounts.toList().joinToString("") { """ --mount type=bind,source=${it.first},target=${it.second} """ } +
                        service.ports.makeMapStr("-p", ":") +
                        service.environment.makeMapStr("-e", "=", true) +
                        service.fixedLabels.makeMapStr("-l", "=") +
                        service.sysCtl.makeMapStr("--sysctl", "=") +
                        labels?.makeMapStr("-l", "=") +
                        " " + service.fullImageName +
                        // After image name MUST BE COMMAND OR NOTHING (including spaces)!!!
                        (command accept """ -c "$command"""")
               ).id

    fun startContainer(id : String, timeOut : Duration? = null) =
        cmdLine("docker start ${timeOut accept "--attach"} $id", timeout = timeOut ?: 1.min)

    fun cleanUp(period : Duration = 1.w) =
        cmdLine("docker system prune -a -f --filter until=${period.str - " "}")

    fun login(registry : DockerRegistry) =
        if (registry.login.isNotBlank())
            cmdLine("docker login ${registry.url} -u ${registry.login} -p ${registry.password}")
        else
            ""

    fun buildImage(dockerFile : File, tag : String, label : String? = null, workDir : File = dockerFile.parentFile) =
        cmdLine("""docker build ./ -f ${dockerFile.canonicalFile.relativeTo(workDir)} ${"--label " and label} -t $tag""", workDir)

    fun removeImage(id : String) =
        cmdLine("docker rmi $id --force")

    fun tag(source : Image, target : Image) =
        cmdLine("docker tag $source $target")

    fun push(tag : String) =
        cmdLine("docker push $tag", timeout = 5.min)

    fun pullImage(fullImageName : String) =
        cmdLine("docker pull $fullImageName", timeout = 15.min)

    fun removeContainer(id : String) =
        cmdLine("docker rm $id --force")

    fun stopContainer(id : String) =
        cmdLine("docker stop $id")

    fun renameContainer(actual : ContainerInstance, newName : String) =
        cmdLine("""docker rename ${actual.service.name} $newName""")

    fun putToContainer(id : String, hostFile : File, containerPath : String) =
        cmdLine("""docker cp ${hostFile.canonicalPath} $id:$containerPath""")

    fun getFromContainer(id : String, hostFile : File, containerPath : String) =
        cmdLine("""docker cp $id:$containerPath ${hostFile.canonicalPath}""")

    fun commitContainer(id : String, image : Image) =
        cmdLine("""docker commit $id ${image.fullName}""")

    fun restartContainer(id : String) =
        cmdLine("""docker restart $id""")

    fun getOutput(id : String) =
        cmdLine("""docker logs $id""", timeout = 5.sec, silent = true)

    fun getServiceOutput(id : String) =
        cmdLine("""docker service logs $id""", timeout = 5.sec, silent = true)

    fun enumItems(cmd : String) =
        cmdLine("docker $cmd", timeout = 5.sec)
            .lines()
            .map { it.trim() }
            .filter { it.isNotEmpty() }
            .map { it.split(divider).map { it.trim('"', ' ') } }

    private infix fun String.deQuotedPairBy(divider : String) =
        substringBefore(divider).trim('"', '\'') to substringAfter(divider).trim('"', '\'')

    private val String.extractParams
        get() = this
            .split(",", "\n")
            .map { it.trim() }
            .filter { it.isNotBlank() }
            .associate { it deQuotedPairBy "=" }

    private fun <K, V> Map<K, V>.makeMapStr(prefix : String, divider : String, mask : Boolean = false) =
        this.toList().joinToString("") {
            """ $prefix ${it.first}$divider${
                if (mask)
                    "'${it.second}'"
                else
                    it.second.str
            } """
        }

    private fun <K, V> Map<K, V>.makeMountStr(suffix : String) =
        this.toList().joinToString("") { """ --mount$suffix type=bind,source=${it.first},target=${it.second} """ }

    private fun <K, V> Map<K, V>.makeMountRmStr() =
        this.toList().joinToString("") { """ --mount-rm ${it.second} """ }

    private fun List<String>.makeListStr(suffix : String) =
        this.joinToString("") { """ $suffix $it """ }

    private val Service.updateParamStr
        get() = if (updateDelay in listOf("", "0", "0s"))
            " --update-parallelism 0 --update-delay 0s "
        else
            " --update-parallelism 1 --update-delay $updateDelay "

    fun createService(service : Service) : String {
        val compatibleService = service.compatible(version)

        return cmdLine("docker service create " +
                               " --with-registry-auth" +
                               " --name ${compatibleService.name} " +
                               " --replicas ${compatibleService.scale} " +
                               compatibleService.mounts.makeMountStr("") +
                               compatibleService.ports.makeMapStr("-p", ":") +
                               compatibleService.environment.makeMapStr("--env", "=", true) +
                               compatibleService.fixedLabels.makeMapStr("-l", "=") +
                               compatibleService.secrets.makeListStr("--secret") +
                               compatibleService.logging.asParams +
                               compatibleService.healthCheck.asParams +
                               compatibleService.sysCtl.makeMapStr("--sysctl", "=") +
                               compatibleService.networks.makeListStr("--network") +
                               compatibleService.updateParamStr +
                               " " + compatibleService.fullImageName +
                               compatibleService.args.joinToString(" ", " "))
    }

    private fun <K, V> update(actual : Map<K, V>, target : Map<K, V>, action : String, divider : String, mask : Boolean = false) =
        actual.diff(target).keys.joinToString("") { " --$action-rm $it " } +
                (target - actual).makeMapStr("--$action-add", divider, mask)

    fun updateService(instance : ServiceInstance, service : Service) : String {
        val compatibleActual = instance.service.compatible(version)
        val compatibleTarget = service.compatible(version)

        return cmdLine("docker service update " +
                               ((compatibleActual.fullImageName != compatibleTarget.fullImageName) accept " --image " + service.fullImageName) +
                               compatibleActual.mounts.filter { it.value !in compatibleTarget.mounts.values }.makeMountRmStr() +
                               compatibleTarget.mounts.makeMountStr("-add") +
                               (compatibleActual.networks - compatibleTarget.networks.toSet()).makeListStr("--network-rm") +
                               (compatibleTarget.networks - compatibleActual.networks.toSet()).makeListStr("--network-add") +
                               (compatibleActual.secrets - compatibleTarget.secrets.toSet()).makeListStr("--secret-rm") +
                               (compatibleTarget.secrets - compatibleActual.secrets.toSet()).makeListStr("--secret-add") +
                               compatibleTarget.logging.asParams +
                               compatibleTarget.healthCheck.asParams +
                               update(compatibleActual.ports, compatibleTarget.ports, "publish", ":") +
                               update(compatibleActual.sysCtl, compatibleTarget.sysCtl, "sysctl", "=") +
                               update(compatibleActual.environment, compatibleTarget.environment, "env", "=", true) +
                               update(compatibleActual.fixedLabels, compatibleTarget.fixedLabels, "label", "=") +
                               compatibleTarget.updateParamStr +
                               ((" --args=\"'" and compatibleTarget.args.joinToString(" ") { it.replace("\"", "\\\"") }) merge "'\" ") +
                               " --replicas ${compatibleTarget.scale} " +
                               " " + compatibleTarget.name,
                       timeout = compatibleTarget.estimatedMaintenanceDuration)
    }

    fun removeService(id : String) =
        cmdLine("docker service rm $id", timeout = 10.sec)

    fun scaleService(id : String, count : Int) =
        cmdLine("docker service scale $id=$count", timeout = 10.sec)

    fun restartService(id : String) =
        cmdLine("docker service update --force $id")

    fun createStack(service : Service) : String {
        val compatibleService = service.compatible(version)

        var res = ""

        fun makeParamList(prefix : String, list : ParamList) =
            "    $prefix:\n" and list.entries.joinToString("\n")
            {
                "      - ${it.key}=${it.value}"
            }.n

        fun makeList(prefix : String, list : List<String>) =
            "    $prefix:\n" and list.joinToString("\n")
            {
                "      - $it"
            }.n

        useTmpFile {
            val command = "version: '3.9'\n" +
                    "services:\n" +
                    "  ${compatibleService.name}:\n" +
                    "    image: ${compatibleService.fullImageName}\n" +
                    "    deploy:\n" +
                    "      replicas: ${compatibleService.scale}\n" +
                    makeParamList("labels", compatibleService.fixedLabels) +
                    makeParamList("volumes", compatibleService.mounts) +
                    makeParamList("environment", compatibleService.environment) +
                    makeParamList("sysctls", compatibleService.sysCtl) +
                    makeList("networks", compatibleService.networks) +
                    (compatibleService.networks.isNotEmpty() accept ("networks:\n" +
                            compatibleService.networks.joinToString("\n") { network -> "  $network:\n    external: true\n" }))

            it.writeText(command)

            res = cmdLine("docker stack deploy --compose-file=$it ${compatibleService.stack}")
        }

        return res
    }

    fun createNetwork(network : k.docker.models.Network, labels : ParamList) =
        cmdLine("docker network create" +
                        (network.attachable accept " --attachable") +
                        " --gateway=" and network.gateway +
                        " --subnet=" and network.subnet +
                        " --scope=" and network.scope +
                        " --driver=" and network.driver +
                        labels.makeMapStr("-label", "=") +
                        network.opt.makeMapStr("-opt", "=") +
                        " ${network.name}",
                timeout = 10.sec)

    fun cleanUpNetworks() =
        cmdLine("docker network prune --force")

    fun removeNetwork(id : String) =
        cmdLine("docker network rm $id")

    fun removeStack(id : String) =
        cmdLine("docker stack rm $id --force", timeout = 10.sec)

    fun createSecret(name : String, content : String, generation : Long) : String {
        val actualGeneration = mute {
            val inspectStr = cmdLine("docker secret inspect $name")

            Regex("\"$kLibServiceGenerationLabel\":\\s*\"(.*?)\"")
                .find(inspectStr)
                ?.groups
                ?.get(1)
                ?.value
                ?.long
                ?: Long.MAX_VALUE
        } ?: -1

        if (actualGeneration >= generation)
            return ""

        if (actualGeneration >= 0)
            cmdLine("docker secret rm $name")

        val res = useTmpFile { contentFile ->
            contentFile.writeText(content)
            cmdLine("docker secret create --label $kLibServiceGenerationLabel=$generation $name $contentFile")
        }

        println(res)

        return res
    }

    fun removeSecret(name : String) =
        cmdLine("docker secret rm $name")

    private fun dockerCall(registry : DockerRegistry, path : String, transformer : (String) -> List<String>) : List<String> {
        /*val client = if (registry.password.isNotBlank())
            ClientFilters.BasicAuth(registry.login, registry.password).then(httpClient)
        else
            httpClient

        val roots = listOf("https", "http")

        return tryProc(10.sec, 2.sec) {
            roots
                .firstNotNullOf { root ->
                    val request = Request(Method.GET, "$root://${registry.url}/$path")
                    val response = client(request)

                    if (response.status == Status.NOT_FOUND)
                        listOf()
                    else if (response.status != Status.OK) {
                        if (root == roots.last())
                            error("Failed to docker call [${response.status}] with request ${request.toCurl()}")
                        else
                            null
                    }
                    else
                        transformer(response.bodyString())
                }
        }*/

        return listOf()
    }

    fun images(registry : DockerRegistry) =
        dockerCall(registry, "v2/_catalog") { response ->
            response
                .deSerialize<ImagesDesc>()
                .repositories
        }

    fun imageVersions(registry : DockerRegistry, image : String) =
        dockerCall(registry, "v2/${image.low}/tags/list") { response ->
            response
                .deSerialize<ImageDesc>()
                .tags
                .sortedWith { value1, value2 ->
                    compareVersions(value2, value1)
                }
        }

    fun compareVersions(version1 : String, version2 : String) : Int {
        val ver1 = version1.substringBefore("-").split('.')
        val ver2 = version2.substringBefore("-").split('.')

        val commonLen = min(ver2.size, ver1.size)

        ver1
            .take(commonLen)
            .forEachIndexed { index, v1 ->
                val diff = mute({ 0 }) { ver2[index].replace("_", "").int - v1.replace("_", "").int }

                if (diff != 0)
                    return diff
            }

        return 0
    }

    fun <T> apiCheck(actualVersion : String, minVersion : String, default : T, code : () -> T) =
        if (compareVersions(minVersion, actualVersion) >= 0)
            code()
        else
            default

    val version by lazy {
        if (isWindows)
            "0.0.0"
        else
            mute {
                tryProc(30.sec) {
                    Regex("""Version":\s*"(.*?)"""")
                        .find(cmdLine("""curl --unix-socket /var/run/docker.sock http://localhost/version 2>/dev/null"""))
                        ?.groups
                        ?.lastOrNull()
                        ?.value
                        ?: "0.0"
                }
            } ?: "0"
    }

    val String.date
        get() = Date(OffsetDateTime.parse(this, DateTimeFormatter.ISO_OFFSET_DATE_TIME).toInstant().toEpochMilli())

    val String.status
        get() = when (val value = this.low.substringBefore(" ")) {
            "up", "running"                                    -> InstanceStatus.Up
            "notexist"                                         -> InstanceStatus.NotExist
            "created"                                          -> InstanceStatus.Created
            "exited", "failed", "complete", "dead", "removing" -> InstanceStatus.Exited
            "restarting"                                       -> InstanceStatus.Restarting

            else                                               -> error("Unknown status [$value]")
        }
}