package k.docker

import k.common.*
import k.docker.Docker.status
import k.docker.models.*
import k.ofl.OFL
import k.serializing.deSerialize
import java.util.stream.Stream

const val swarmServiceLabel = "com.docker.swarm.service.name"
const val swarmStackLabel = "com.docker.stack.namespace"
const val maxCmdParams = 100

interface DockerSnapshot {
    val images : List<ImageInstance>
    val containers : List<ContainerInstance>
    val services : List<ServiceInstance>
    val networks : List<NetworkInstance>
}

class Snapshot(private val source : String = "Snapshot") : DockerSnapshot {
    companion object {
        val EMPTY = object : DockerSnapshot {
            override val images = listOf<ImageInstance>()
            override val containers = listOf<ContainerInstance>()
            override val services = listOf<ServiceInstance>()
            override val networks = listOf<NetworkInstance>()
        }
    }

    override val networks by lazy {
        enum<NetworkParams>("network ls --no-trunc")
            .map { params ->
                val ipamConfig = params.IPAM.Config.firstOrNull()

                NetworkInstance(params.Id,
                                k.docker.models.Network(
                                        params.Name,
                                        params.Attachable,
                                        params.Scope,
                                        params.Driver,
                                        ipamConfig?.Subnet ?: "",
                                        ipamConfig?.Gateway ?: "",
                                        params.Options,
                                        params.Labels),
                                params.Containers.values.map { it.Name })
            }.toList()
    }

    override val images by lazy {
        enum<ImageParams>("image ls --no-trunc")
            .map { params ->
                val fullImageName = if (params.RepoTags.isNotEmpty())
                    params.RepoTags.first()
                else if (params.RepoDigests.isNotEmpty())
                    params.RepoDigests.first().substringBefore("@")
                else
                    "unknown"

                ImageInstance(
                        params.Id.substringAfter(":"),
                        fullImageName.image,
                        params.Created,
                        params.Config.Labels)
            }.toList()
    }

    override val containers by lazy {
        enum<ContainerParams>("ps -a --no-trunc")
            .map { params ->
                val labels = params.Config.Labels
                val containerName = params.Name.drop(1)
                val serviceName = labels[swarmServiceLabel] ?: containerName
                val image = images.find { it.id == params.Image.substringAfter(":") }?.image
                val stack = labels[swarmStackLabel] ?: ""

                val service = Service(
                        serviceName - (stack merge "_"),
                        image?.registry ?: "",
                        image?.name ?: "",
                        image?.version ?: "",
                        params.Mounts.associate { it.Destination pairBy "=" },
                        params.HostConfig.PortBindings.entries
                            .associate { it.key.substringBefore("/") to (it.value.extract("HostPort\"\\s*:\\s*\"(\\d*)") default "0") },
                        params.Config.Env.associate { it pairBy "=" },
                        labels,
                        mapOf(),
                        params.NetworkSettings.Networks.keys.toList(),
                        "1",
                        "0s",
                        stack,
                        listOf(),
                        Logging(),
                        HealthCheck(),
                        mapOf(),
                        OFL.EMPTY,
                        ServiceType.Container,
                        source = source)

                ContainerInstance(
                        params.Id,
                        service,
                        params.Created,
                        params.State.Status.status,
                        if (params.State.Status.status == InstanceStatus.Exited)
                            Duration(params.State.FinishedAt)
                        else
                            Duration(params.State.StartedAt),
                        params.State.ExitCode)
            }.toList()
    }

    override val services by lazy {
        enum<ServiceParams>("service ls")
            .toList()
            .map { params ->
                val image = params.Spec.TaskTemplate.ContainerSpec.Image.substringBefore("@").image
                val labels = params.Spec.Labels
                val stack = labels[swarmStackLabel] ?: ""
                val serviceNetworks = (networks.filter { it.id in params.Spec.TaskTemplate.Networks.map { network -> network.Target } }).map { it.network.name }

                val serviceContainers = containers.filter { it.service.fullName same params.Spec.Name }

                val networks = serviceNetworks.ifEmpty {
                    serviceContainers.find { it.status == InstanceStatus.Up }?.service?.networks
                        ?: serviceContainers.firstOrNull()?.service?.networks
                        ?: listOf()
                }

                val logDriver = params.Spec.TaskTemplate.LogDriver
                val healthCheck = params.Spec.TaskTemplate.ContainerSpec.Healthcheck

                val service = Service(
                        params.Spec.Name - (stack merge "_"),
                        image.registry,
                        image.name,
                        image.version,
                        params.Spec.TaskTemplate.ContainerSpec.Mounts.associate { it.Source to it.Target },
                        params.Endpoint.Ports.associate { it.PublishedPort.str to it.TargetPort.str },
                        params.Spec.TaskTemplate.ContainerSpec.Env.associate { it pairBy "=" },
                        labels,
                        params.Spec.TaskTemplate.ContainerSpec.Sysctls,
                        networks,
                        params.Spec.Mode.Replicated.Replicas.str,
                        if (params.Spec.UpdateConfig.Delay == 0L) "0s" else Duration(params.Spec.UpdateConfig.Delay).str - " ",
                        stack,
                        params.Spec.TaskTemplate.ContainerSpec.Secrets.map { it.SecretName },
                        Logging(
                                logDriver.Options.maxFile.ifBlank { "0" }.int,
                                logDriver.Name,
                                logDriver.Options.maxSize).normalized,
                        HealthCheck(
                                healthCheck.Test.lastOrNull() ?: "",
                                Duration(healthCheck.StartPeriod).str - " ",
                                healthCheck.Retries,
                                Duration(healthCheck.Interval).str - " ",
                                Duration(healthCheck.Timeout).str - " ").normalized,
                        mapOf(),
                        OFL.EMPTY,
                        if (stack.isNotBlank())
                            ServiceType.Stack
                        else
                            ServiceType.Service,
                        args = params.Spec.TaskTemplate.ContainerSpec.Args,
                        generation = labels[kLibServiceGenerationLabel]?.long ?: 0,
                        source = source)

                val actualCount = serviceContainers.count { it.status.online }

                val status = if (params.Spec.Mode.Replicated.Replicas == 0)
                    InstanceStatus.Exited
                else if (actualCount < params.Spec.Mode.Replicated.Replicas)
                    InstanceStatus.Restarting
                else
                    InstanceStatus.Up

                ServiceInstance(
                        params.ID,
                        service,
                        params.CreatedAt,
                        status,
                        actualCount,
                        Duration(params.UpdatedAt),
                        serviceContainers)
            }
            .toList() + containers
            .filter { it.service.labels[swarmServiceLabel].isNull }
            .map {
                ServiceInstance(
                        it.id,
                        it.service,
                        it.created,
                        it.status,
                        if (it.status.online) 1 else 0,
                        it.statusDuration,
                        listOf(it))
            }
    }

    fun service(id : String) =
        services.find { it.service.name same id || it.id == id } mustBeFound id

    private inline fun <reified T : Any> enum(cmd : String) : Stream<T> {
        return cmdLine("docker $cmd --format {{.ID}}", timeout = 10.sec)
            .lines()
            .map { it.trim() }
            .filter { it.isNotEmpty() }
            .chunked(maxCmdParams)
            .parallelStream()
            .flatMap { chunk ->
                val idList = chunk.joinToString(" ")

                (cmdLine("docker inspect $idList --format json", timeout = 10.sec, silent = true).substringBeforeLast("}]") + "}]")
                    .deSerialize<List<T>>()
                    .parallelStream()
            }
    }

    private val String.params
        get() = this
            .split(",")
            .map { it.trim() }
            .filter { it.isNotEmpty() }
            .associate { it pairBy "=" }
}