package k.docker.models

import k.common.*
import k.docker.Docker
import k.extensions.fmt
import k.serializing.Optional
import k.ofl.*

const val kLibServicePrefix = "k.lib.service"
const val kLibServiceGenerationLabel = "$kLibServicePrefix.generation"
val minimumMaintenanceDuration = 30.sec

data class Service(val name : String,
                   @Optional
                   val registry : String,
                   @Optional
                   val image : String,
                   @Optional
                   val version : String,
                   @Optional
                   val mounts : ParamList = mapOf(),
                   @Optional
                   val ports : ParamList = mapOf(),
                   @Optional
                   val environment : ParamList = mapOf(),
                   @Optional
                   val labels : ParamList = mapOf(),
                   @Optional
                   val sysCtl : ParamList = mapOf(),
                   @Optional
                   val networks : List<String> = listOf(),
                   @Optional("1")
                   val scale : String = "1",
                   @Optional("0s")
                   val updateDelay : String = "0s",
                   @Optional
                   val stack : String = "",
                   @Optional
                   val secrets : List<String> = listOf(),
                   @Optional
                   val logging : Logging = Logging(),
                   @Optional
                   val healthCheck : HealthCheck = HealthCheck(),
                   @Optional
                   val checks : Map<String, CheckRule> = mapOf(),
                   @Optional
                   val proxy : OflObject = OFL.EMPTY,
                   @Optional("Service")
                   val type : ServiceType = ServiceType.Service,
                   @Optional
                   val configuration : OflObject = OFL.EMPTY,
                   @Optional
                   val files : Map<String, String> = mapOf(),
                   @Optional
                   val args : List<String> = listOf(),
                   @Optional("0")
                   val generation : Long = 0,
                   @Optional("Create")
                   val action : String = "Create",
                   @Optional
                   val source : String = "",
                   @Optional
                   val configSource : OflObject = OFL.EMPTY,
                   @Optional("0")
                   val priority : Int = 0,
                   @Optional
                   val role : String = "",
                   @Optional("false")
                   val hidden : Boolean = false) {
    val fullImageName
        get() = "${registry.str merge "/"}$image${":" and version}"

    val fullName
        get() = (stack merge "_") + name

    val fullContainerName
        get() = "$fullName($image)"

    val fixedLabels
        get() = if (generation.isZero) labels else labels + (kLibServiceGenerationLabel to generation.str)

    val estimatedMaintenanceDuration
        get() = Duration(updateDelay) * scale.int + minimumMaintenanceDuration

    fun compatible(version : String) =
        copy(sysCtl = Docker.apiCheck(version, "19.3", mapOf()) { sysCtl })

    val asFullScript
        get() = cleanUpScript(copy(generation = 0,
                                   source = "",
                                   role = "",
                                   hidden = false,
                                   labels = labels.filter { it.key != kLibServiceGenerationLabel })
                                  .fmt)

    val asScript
        get() = cleanUpScript(copy(generation = 0,
                                   source = "",
                                   role = "",
                                   hidden = false,
                                   configuration = OFL.EMPTY,
                                   checks = mapOf(),
                                   files = mapOf(),
                                   priority = 0,
                                   labels = labels.filter { it.key != kLibServiceGenerationLabel },
                                   networks = networks.filter { it != "ingress" && it != "bridge" && it != "my-ingress" })
                                  .fmt)
}

fun cleanUpScript(script : String) =
    script
        .replace("""\w+\s*\{\}""".toRegex(), "")
        .let { content ->
            OFL(content).fmt
        }