/*
 * Copyright © 2016 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ru.tinkoff.acquiring.sdk;

import android.app.Activity;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.JavascriptInterface;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import java.net.URLEncoder;

import ru.tinkoff.acquiring.sdk.responses.GetAddCardStateResponse;

/**
 * @author a.shishkin1
 */
public class ThreeDsFragment extends Fragment {

    public static final String EXTRA_3DS = "extra_3ds";

    private static final String CANCEL_ACTION = "cancel.do";
    private static final String SUBMIT_3DS_AUTHORIZATION = "Submit3DSAuthorization";

    private WebView wvThreeDs;
    private ThreeDsData data;
    private AcquiringSdk sdk;
    private String termUrl;

    public static ThreeDsFragment newInstance(Bundle threeDSBundle) {
        ThreeDsFragment fragment = new ThreeDsFragment();
        Bundle args = new Bundle();
        args.putBundle(EXTRA_3DS, threeDSBundle);
        fragment.setArguments(args);
        return fragment;
    }

    @JavascriptInterface
    public void logHtml(String msg) {
        Journal.log(msg);
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.acq_fragment_3ds, container, false);
        wvThreeDs = (WebView) view.findViewById(R.id.wv_3ds);
        wvThreeDs.setWebViewClient(new ThisWebViewClient());
        wvThreeDs.getSettings().setDomStorageEnabled(true);
        wvThreeDs.getSettings().setJavaScriptEnabled(true);
        wvThreeDs.getSettings().setJavaScriptCanOpenWindowsAutomatically(true);
        wvThreeDs.addJavascriptInterface(this, "LogPage");
        data = new ThreeDsBundlePacker().unpack(getArguments().getBundle(EXTRA_3DS));
        return view;
    }

    @Override
    public void onActivityCreated(Bundle savedInstanceState) {
        super.onActivityCreated(savedInstanceState);
        sdk = ((IBaseSdkActivity) getActivity()).getSdk();
        String url = data.getAcsUrl();
        termUrl = sdk.getUrl(SUBMIT_3DS_AUTHORIZATION) + "/" + SUBMIT_3DS_AUTHORIZATION;
        try {
            String params = new StringBuilder()
                    .append("PaReq=").append(URLEncoder.encode(data.getPaReq(), "UTF-8"))
                    .append("&MD=").append(URLEncoder.encode(data.getMd(), "UTF-8"))
                    .append("&TermUrl=").append(URLEncoder.encode(termUrl, "UTF-8"))
                    .toString();
            wvThreeDs.postUrl(url, params.getBytes());
        } catch (Exception e) {
            throw new AcquiringSdkException(e);
        }
    }

    private class ThisWebViewClient extends WebViewClient {

        boolean canceled = false;

        @Override
        public void onPageFinished(WebView view, String url) {
            AcquiringSdk.log("webview redirect: " + url);
            super.onPageFinished(view, url);

            if (url != null && url.startsWith("http")) {
                view.loadUrl("javascript:LogPage.logHtml(document.getElementsByTagName('html')[0].innerHTML);");
            }

            if (url.contains(CANCEL_ACTION)) {
                canceled = true;
                Activity activity = (Activity) view.getContext();
                activity.setResult(Activity.RESULT_CANCELED);
                activity.finish();
            }

            if (termUrl.equals(url)) {
                view.setVisibility(View.INVISIBLE);
                if (!canceled) {
                    requestState(sdk, data);
                }
            }
        }

        @Override
        public boolean shouldOverrideUrlLoading(WebView view, String url) {
            return super.shouldOverrideUrlLoading(view, url);
        }
    }

    private static void requestState(final AcquiringSdk sdk, final ThreeDsData threeDsData) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    if (threeDsData.isPayment()) {
                        PaymentStatus status = sdk.getState(threeDsData.getPaymentId());
                        if (status == PaymentStatus.CONFIRMED || status == PaymentStatus.AUTHORIZED) {
                            CommonSdkHandler.INSTANCE.obtainMessage(CommonSdkHandler.SUCCESS).sendToTarget();
                        } else {
                            CommonSdkHandler.INSTANCE.obtainMessage(CommonSdkHandler.EXCEPTION, new AcquiringSdkException(new IllegalStateException("PaymentState = " + status))).sendToTarget();
                        }
                    } else {
                        GetAddCardStateResponse response = sdk.getAddCardState(threeDsData.getRequestKey());
                        PaymentStatus status = response.getStatus();
                        if (status == PaymentStatus.COMPLETED) {
                            AttachCardFormHandler.INSTANCE.obtainMessage(AttachCardFormHandler.CARD_ID, response.getCardId()).sendToTarget();
                            CommonSdkHandler.INSTANCE.obtainMessage(CommonSdkHandler.SUCCESS).sendToTarget();
                        } else {
                            CommonSdkHandler.INSTANCE.obtainMessage(CommonSdkHandler.EXCEPTION, new AcquiringSdkException(new IllegalStateException("PaymentState = " + status))).sendToTarget();
                        }
                    }
                } catch (Exception e) {
                    CommonSdkHandler.INSTANCE.obtainMessage(CommonSdkHandler.EXCEPTION, e).sendToTarget();
                }
            }
        }).start();
    }
}
