package ru.tinkoff.acquiring.sdk.cardscanners.ui

import android.app.AlertDialog
import android.content.Intent
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Build
import android.os.Bundle
import androidx.fragment.app.Fragment
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import androidx.activity.addCallback
import androidx.core.content.ContextCompat
import androidx.core.os.bundleOf
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.cardscanners.models.AsdkScannedCardData
import ru.tinkoff.acquiring.sdk.databinding.FragmentAsdkNfcScanBinding
import ru.tinkoff.core.components.nfc.NfcHelper
import ru.tinkoff.core.components.nfc.NfcUtils

class AsdkNfcScanFragment : Fragment() {

    private lateinit var nfcHelper: NfcHelper

    private var binding: FragmentAsdkNfcScanBinding? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return FragmentAsdkNfcScanBinding.inflate(inflater, container, false)
            .also { this.binding = it }
            .root
    }

    override fun onDestroyView() {
        super.onDestroyView()
        binding = null
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            setResult(AsdkNfcScanResult.BackPressed)
        }

        nfcHelper = NfcHelper.createAndRegisterObserver(requireActivity(), object : NfcHelper.Callback {
            override fun onResult(bundle: Bundle) =
                onResult(bundle.getString(NfcHelper.CARD_NUMBER)!!, bundle.getString(NfcHelper.EXPIRY_DATE)!!)

            override fun onException(p0: java.lang.Exception?) = onException()

            override fun onNfcNotSupported() = onException()

            override fun onNfcDisabled() = showDialog()
        })

        setupTranslucentStatusBar()

        if(binding == null) return

        binding?.acqNfcTvDescription?.setText(R.string.acq_scan_by_nfc_description)

        with(binding!!.acqNfcBtnClose) {
            setText(R.string.acq_scan_by_nfc_close)
            setOnClickListener { setResult(AsdkNfcScanResult.Close) }
        }

        applyBackgroundColor()
    }

    private fun setResult(result: AsdkNfcScanResult) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    private fun onResult(cardNumber: String, expireDate: String) {
        val card = AsdkScannedCardData(cardNumber, expireDate, "")
        val intent = Intent()
        intent.putExtra(EXTRA_CARD, card)
        setResult(AsdkNfcScanResult.Finish(intent))
    }

    private fun setupTranslucentStatusBar() {
        if (Build.VERSION.SDK_INT in 19..20) {
            requireActivity().window.addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
        }
        if (Build.VERSION.SDK_INT >= 21) {
            requireActivity().window.statusBarColor = Color.TRANSPARENT
        }
    }

    private fun applyBackgroundColor() {
        val rootView = binding?.acqNfcLlRoot
        val currentBackground = rootView?.background as ColorDrawable
        val currentColor = currentBackground.color

        if (currentColor == ContextCompat.getColor(requireContext(), R.color.acq_colorNfcBackground)) {
            val newColor = ALPHA_MASK and currentColor
            rootView.setBackgroundColor(newColor)
        }
    }

    private fun onException() {
        setResult(AsdkNfcScanResult.Error)
    }

    private fun showDialog() {
        AlertDialog.Builder(requireContext())
            .setTitle(getString(R.string.acq_nfc_is_disable))
            .setMessage(getString(R.string.acq_nfc_is_disable))
            .setPositiveButton(android.R.string.ok) { _, _ ->
                NfcUtils.openNfcSettingsForResult(requireActivity(), REQUEST_CODE_SETTINGS)
            }.setNegativeButton(android.R.string.cancel) { _, _ -> setResult(AsdkNfcScanResult.Close) }
            .show()
    }

    companion object {

        const val EXTRA_CARD = "card_extra"
        const val RESULT_ERROR = 256

        const val REQUEST_CODE_SETTINGS = 0

        private const val ALPHA_MASK = -0x33000001

        private val TAG = this::class.simpleName
        internal val FRAGMENT_RESULT_KEY = "$TAG.FRAGMENT_RESULT_KEY"
        internal val FRAGMENT_RESULT_BUNDLE_KEY = "$TAG.FRAGMENT_RESULT_BUNDLE_KEY"
    }
}