package ru.tinkoff.acquiring.sdk.payment

import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancelChildren
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkTimeoutException
import ru.tinkoff.acquiring.sdk.exceptions.getErrorCodeIfApiError
import ru.tinkoff.acquiring.sdk.models.enums.ResponseStatus
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.payment.methods.MirPayMethods
import ru.tinkoff.acquiring.sdk.payment.methods.MirPayMethodsImpl
import ru.tinkoff.acquiring.sdk.payment.pooling.GetStatusMethod
import ru.tinkoff.acquiring.sdk.payment.pooling.GetStatusPooling

/**
 * @author k.shpakovskiy
 */
class MirPayProcess internal constructor(
    private val getStatusPooling: GetStatusPooling,
    private val linkMethods: MirPayMethods,
    private val scope: CoroutineScope
) {

    internal constructor(
        sdk: AcquiringSdk,
        ioDispatcher: CoroutineDispatcher = Dispatchers.IO
    ) : this(
        GetStatusPooling(GetStatusMethod.Impl(sdk)),
        MirPayMethodsImpl(sdk),
        CoroutineScope(ioDispatcher)
    )

    private val _stateFlow = MutableStateFlow<MirPayPaymentState>(MirPayPaymentState.Created(null))
    val stateFlow = _stateFlow.asStateFlow()
    private var looperJob: Job = Job()

    fun start(paymentOptions: PaymentOptions) {
        scope.launch {
            runCatching { startFlow(paymentOptions) }
                .onFailure { handlePaymentFlowFailure(it) }
        }
    }

    fun goingToBankApp() {
        when (val currentState = _stateFlow.value) {
            is MirPayPaymentState.Stopped,
            is MirPayPaymentState.NeedChooseOnUi -> {
                setState(MirPayPaymentState.LeaveOnBankApp(currentState.paymentId!!))
            }
            else -> Unit
        }
    }

    fun startCheckingStatus() {
        val currentState = _stateFlow.value
        val paymentId = currentState.paymentId ?: return
        if (currentState is MirPayPaymentState.LeaveOnBankApp || currentState is MirPayPaymentState.Paused) {
            setState(MirPayPaymentState.CheckingStatus(paymentId))
            looperJob.cancel()
            looperJob = startLoping(paymentId)
        }
    }

    fun pause() {
        val currentState = _stateFlow.value
        if (currentState is MirPayPaymentState.CheckingStatus) {
            setState(MirPayPaymentState.Paused(currentState.paymentId))
            if (looperJob.isActive) {
                looperJob.cancel()
            }
        }
    }

    fun stop() {
        setState(MirPayPaymentState.Stopped(_stateFlow.value.paymentId))
        if (scope.isActive) {
            scope.coroutineContext.cancelChildren()
        }
    }

    private suspend fun startFlow(paymentOptions: PaymentOptions) {
        val paymentId = paymentOptions.paymentId ?: linkMethods.init(paymentOptions)
        setState(MirPayPaymentState.Started(paymentId))
        val link = linkMethods.getLink(paymentId)
        setState(MirPayPaymentState.NeedChooseOnUi(paymentId, link))
    }


    private fun setState(newState: MirPayPaymentState) {
        _stateFlow.value = newState
    }

    private fun handlePaymentFlowFailure(ex: Throwable) {
        if (ex is CancellationException)
            return
        ex as Exception
        setState(MirPayPaymentState.PaymentFailed(_stateFlow.value.paymentId, ex, ex.getErrorCodeIfApiError()))
    }

    private fun startLoping(paymentId: Long): Job {
        return scope.launch {
            getStatusPooling.start(
                retriesCount = POLLING_RETRIES_COUNT,
                paymentId = paymentId,
                delayMs = POLLING_DELAY_MS
            )
                .map {
                    mapResponseStatusToState(status = it, paymentId = paymentId)
                }
                .catch {
                    emit(MirPayPaymentState.PaymentFailed(throwable = it, paymentId = paymentId))
                }
                .collectLatest { setState(it) }
        }
    }

    private fun mapResponseStatusToState(status: ResponseStatus, paymentId: Long) = when (status) {
        ResponseStatus.AUTHORIZED,
        ResponseStatus.CONFIRMED -> { MirPayPaymentState.Success(paymentId) }
        ResponseStatus.REJECTED -> {
            MirPayPaymentState.PaymentFailed(
                paymentId,
                AcquiringSdkException(IllegalStateException("PaymentState = $status"), paymentId = paymentId)
            )
        }
        ResponseStatus.DEADLINE_EXPIRED -> {
            MirPayPaymentState.PaymentFailed(
                paymentId,
                AcquiringSdkTimeoutException(IllegalStateException("PaymentState = $status"))
            )
        }
        else -> MirPayPaymentState.CheckingStatus(paymentId, status)
    }

    companion object {
        private const val POLLING_DELAY_MS: Long = 5000
        private const val POLLING_RETRIES_COUNT: Int = 60
    }
}

sealed interface MirPayPaymentState {
    val paymentId: Long?

    class Created(
        override val paymentId: Long? = null
    ) : MirPayPaymentState

    class Started(
        override val paymentId: Long
    ) : MirPayPaymentState

    class NeedChooseOnUi(
        override val paymentId: Long,
        val deeplink: String
    ) : MirPayPaymentState

    class Success(
        override val paymentId: Long
    ) : MirPayPaymentState

    class LeaveOnBankApp(
        override val paymentId: Long
    ) : MirPayPaymentState

    class CheckingStatus(
        override val paymentId: Long,
        val status: ResponseStatus? = null
    ) :
        MirPayPaymentState

    class Paused(
        override val paymentId: Long
    ) : MirPayPaymentState

    class PaymentFailed(
        override val paymentId: Long?,
        val throwable: Throwable,
        val errorCode: String? = null
    ) : MirPayPaymentState

    class Stopped(
        override val paymentId: Long?
    ) : MirPayPaymentState
}
