package ru.tinkoff.acquiring.sdk.payment

import android.app.Application
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.NonCancellable
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringApiException
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.paysources.AttachedCard
import ru.tinkoff.acquiring.sdk.models.result.PaymentResult
import ru.tinkoff.acquiring.sdk.network.AcquiringApi
import ru.tinkoff.acquiring.sdk.payment.methods.ChargeMethods
import ru.tinkoff.acquiring.sdk.payment.methods.ChargeMethodsSdkImpl
import ru.tinkoff.acquiring.sdk.payment.methods.Check3DsVersionMethods
import ru.tinkoff.acquiring.sdk.payment.methods.Check3DsVersionMethodsSdkImpl
import ru.tinkoff.acquiring.sdk.payment.methods.FinishAuthorizeMethods
import ru.tinkoff.acquiring.sdk.payment.methods.FinishAuthorizeMethodsSdkImpl
import ru.tinkoff.acquiring.sdk.payment.methods.InitMethods
import ru.tinkoff.acquiring.sdk.payment.methods.InitMethodsSdkImpl
import ru.tinkoff.acquiring.sdk.payment.methods.requiredPaymentId
import ru.tinkoff.acquiring.sdk.payment.pooling.GetStatusMethod
import ru.tinkoff.acquiring.sdk.payment.pooling.GetStatusPooling
import ru.tinkoff.acquiring.sdk.threeds.AppBaseProcessResult
import ru.tinkoff.acquiring.sdk.threeds.CertificateManagerImpl
import ru.tinkoff.acquiring.sdk.threeds.CreateAppBasedTransactionImpl
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBaseProcess
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBaseProcessImpl
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBasedTransaction
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsDataCollector
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsHelper
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.toggles.toggles.AppBaseFeatureToggle
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeResult
import ru.tinkoff.acquiring.sdk.utils.CoroutineManager

/**
 * Created by i.golovachev
 */
class RecurrentPaymentProcess internal constructor(
    private val initMethods: InitMethods,
    private val chargeMethods: ChargeMethods,
    private val check3DsVersionMethods: Check3DsVersionMethods,
    private val finishAuthorizeMethods: FinishAuthorizeMethods,
    private val scope: CoroutineScope = CoroutineScope(SupervisorJob()),
    private val coroutineManager: CoroutineManager = CoroutineManager(), // TODO нужны только диспы отсюда
    private val getStatusPooling: GetStatusPooling,
    private val dsAppBaseProcess: ThreeDsAppBaseProcess,
    private val toggleManager: FeatureToggleManager
) {

    private val _state = MutableStateFlow<PaymentByCardState>(PaymentByCardState.Created)
    private val _paymentIdOrNull get() = (_state.value as? PaymentByCardState.Started)?.paymentId // TODO после слияния состояний
    private val rejectedPaymentId = MutableStateFlow<String?>(null)
    val state = _state.asStateFlow()

    fun start(
        cardData: AttachedCard,
        paymentOptions: PaymentOptions,
        email: String? = null
    ) {
        scope.launch(coroutineManager.io) {
            try {
                startPaymentFlow(cardData, paymentOptions, email)
            } catch (e: Throwable) {
                handleException(paymentOptions, e, _paymentIdOrNull, true)
            }
        }
    }

    fun startWithCvc(
        cvc: String,
        rebillId: String,
        rejectedId: String,
        paymentOptions: PaymentOptions,
        email: String?
    ) {
        check(_state.value is PaymentByCardState.CvcUiNeeded)
        scope.launch(coroutineManager.io) {
            try {
                startRejectedFlow(cvc, rebillId, rejectedId, paymentOptions, email)
            } catch (e: Throwable) {
                handleException(paymentOptions, e, _paymentIdOrNull, false)
            }
        }
    }

    fun recreate() {
        _state.value = PaymentByCardState.Created
    }

    fun onThreeDsUiInProcess() {
        _state.value = PaymentByCardState.ThreeDsInProcess
    }

    fun set3dsResult(paymentResult: PaymentResult, paymentOptions: PaymentOptions) {
        _state.value =
            PaymentByCardState.Success(
                paymentResult.paymentId,
                paymentResult.cardId,
                paymentResult.rebillId,
                amount = paymentOptions.order.amount,
            )
    }

    fun set3dsResult(error: Throwable?, paymentId: Long?) {
        _state.value =
            PaymentByCardState.Error(error ?: AcquiringSdkException(IllegalStateException()), paymentId)
    }

    private suspend fun startPaymentFlow(
        cardData: AttachedCard,
        paymentOptions: PaymentOptions,
        email: String?
    ) {
        val paymentId = paymentOptions.paymentId
            ?: initMethods
                .init(paymentOptions, email)
                .requiredPaymentId()

        _state.value = PaymentByCardState.Started(
            paymentOptions = paymentOptions,
            email = email,
            paymentId = paymentId
        )

        val result = chargeMethods
            .charge(paymentId = paymentId, rebillId = cardData.rebillId)

        when (result) {
            is ChargeMethods.Result.Success -> _state.value = PaymentByCardState.Success(
                paymentId = result.paymentId,
                cardId = result.cardId,
                rebillId = result.rebillId,
                amount = paymentOptions.order.amount,
            )

            is ChargeMethods.Result.Failure -> _state.value = PaymentByCardState.Error(
                throwable = result.failure,
                paymentId = paymentId
            )

            is ChargeMethods.Result.NeedGetState -> {
                startGetStatePolling(result.paymentId, paymentId, paymentOptions)
            }
        }
    }

    private fun startGetStatePolling(
        resultPaymentId: Long,
        paymentId: Long,
        paymentOptions: PaymentOptions
    ) {
        coroutineManager.launchOnBackground {
            getStatusPooling.start(paymentId = resultPaymentId)
                .map {
                    PaymentByCardState.mapResponseStatusToState(
                        status = it,
                        paymentId = paymentId,
                        paymentOptions = paymentOptions
                    )
                }
                .catch {
                    emit(
                        PaymentByCardState.Error(
                            throwable = it,
                            paymentId = paymentId
                        )
                    )
                }
                .collectLatest {
                    _state.value = it
                }
        }
    }

    private suspend fun startRejectedFlow(
        cvc: String,
        rebillId: String,
        rejectedId: String,
        paymentOptions: PaymentOptions,
        email: String?
    ) {
        _state.value = PaymentByCardState.CvcUiInProcess

        val cardId = chargeMethods
            .getCardByRebillId(
                rebillId = rebillId,
                paymentOptions = paymentOptions
            )
            .cardId

        val card = AttachedCard(cardId = cardId, cvc = cvc)

        val paymentId = paymentOptions.paymentId
            ?: chargeMethods
                .init(
                    paymentOptions = paymentOptions,
                    email = email,
                    rejectedPaymentId = rejectedId
                )
                .requiredPaymentId()

        _state.value = PaymentByCardState.Started(paymentOptions, email, paymentId)

        val data3ds = check3DsVersionMethods
            .callCheck3DsVersion(
                paymentId,
                card,
                paymentOptions,
                email
            )

        val isAppBase = data3ds.dsResponse.isAppBase()
        var appBaseFaResult: AppBaseFaResult? = null

        val finish = when {
            isAppBase && toggleManager.isEnabled(AppBaseFeatureToggle) -> {
                appBaseFaResult = checkAppBaseFa(data3ds, paymentId, paymentOptions, email, card)
                appBaseFaResult.finishAuthorizeMethods
            }

            else -> {
                finishAuthorizeMethods.finish(
                    paymentId,
                    card,
                    paymentOptions,
                    email,
                    data3ds.additionalData,
                    data3ds.threeDsVersion,
                )
            }
        }

        _state.value = when (finish) {
            is FinishAuthorizeMethods.Result.Need3ds -> {
                if (appBaseFaResult != null) {
                    PaymentByCardState.ThreeDsAppBase(
                        transaction = requireNotNull(appBaseFaResult.transaction),
                        threeDsData = finish.threeDsState.data,
                        paymentOptions = paymentOptions
                    )
                } else {
                    PaymentByCardState.ThreeDsUiNeeded(
                        finish.threeDsState,
                        paymentOptions
                    )
                }
            }

            is FinishAuthorizeMethods.Result.Success -> PaymentByCardState.Success(
                finish.paymentId,
                card.cardId,
                rebillId,
                amount = paymentOptions.order.amount,
            )

            is FinishAuthorizeMethods.Result.NeedGetState -> {
                startGetStatePolling(finish.paymentId, paymentId, paymentOptions)
                return
            }
        }
    }

    private suspend fun checkAppBaseFa(
        data3ds: Check3DsVersionMethods.Data,
        paymentId: Long,
        options: PaymentOptions,
        email: String?,
        paymentSource: PaymentSource
    ): AppBaseFaResult {
        when (val appBaseResult = dsAppBaseProcess.collectInfo(data3ds.dsResponse)) {

            is AppBaseProcessResult.Error -> {
                throw appBaseResult.throwable
            }

            is AppBaseProcessResult.Success -> {
                return AppBaseFaResult(
                    finishAuthorizeMethods.finish(
                        paymentId = paymentId,
                        paymentSource = paymentSource,
                        paymentOptions = options,
                        email = email,
                        data = appBaseResult.threedsData,
                    ), appBaseResult.transaction
                )
            }
        }
    }

    private suspend fun handleException(
        paymentOptions: PaymentOptions,
        throwable: Throwable,
        paymentId: Long?,
        needCheckRejected: Boolean
    ) {
        if (throwable is CancellationException) return
        withContext(NonCancellable) {
            _state.emit(
                if (needCheckRejected && checkRejectError(throwable)) {
                    PaymentByCardState.CvcUiNeeded(paymentOptions, saveRejectedId())
                } else {
                    PaymentByCardState.Error(throwable, paymentId)
                }
            )
        }
    }

    private fun checkRejectError(it: Throwable): Boolean {
        return it is AcquiringApiException && it.response!!.errorCode == AcquiringApi.API_ERROR_CODE_CHARGE_REJECTED
    }

    private fun saveRejectedId(): String {
        val value = checkNotNull(_paymentIdOrNull?.toString())
        rejectedPaymentId.value = value
        return value
    }

    private fun onAppBased3DsCancelled() {
        _state.value = PaymentByCardState.Cancelled
    }

    fun setChallengeResult(result: AppBaseChallengeResult, paymentOptions: PaymentOptions) {
        when (result) {
            AppBaseChallengeResult.Cancelled -> {
                onAppBased3DsCancelled()
            }

            is AppBaseChallengeResult.TimeOut -> {
                set3dsResult(result.error, result.paymentId)
            }

            is AppBaseChallengeResult.AppBaseChallengeResultError -> {
                set3dsResult(result.error, result.paymentId)
            }

            is AppBaseChallengeResult.Success -> {
                set3dsResult(result.paymentResult, paymentOptions)
            }

            AppBaseChallengeResult.Loading -> {
                onThreeDsUiInProcess()
            }
        }
    }

    companion object {

        private var value: RecurrentPaymentProcess? = null

        @JvmStatic
        fun get() = value!!

        @JvmStatic
        @Synchronized
        fun init(
            sdk: AcquiringSdk,
            application: Application,
            threeDsDataCollector: ThreeDsDataCollector = ThreeDsHelper.CollectData
        ) {
            value = RecurrentPaymentProcess(
                InitMethodsSdkImpl(sdk),
                ChargeMethodsSdkImpl(sdk),
                Check3DsVersionMethodsSdkImpl(sdk, application, threeDsDataCollector, FeatureToggleManager.getInstance(application)),
                FinishAuthorizeMethodsSdkImpl(sdk),
                CoroutineScope(Job()),
                CoroutineManager(),
                GetStatusPooling(GetStatusMethod.Impl(sdk)),
                ThreeDsAppBaseProcessImpl(
                    CreateAppBasedTransactionImpl(
                        CertificateManagerImpl(
                            application
                        ), application
                    )
                ),
                FeatureToggleManager.getInstance(application)
            )
        }
    }

    private class AppBaseFaResult(
        val finishAuthorizeMethods: FinishAuthorizeMethods.Result,
        val transaction: ThreeDsAppBasedTransaction? = null
    )
}
