package ru.tinkoff.acquiring.sdk.payment

import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkTimeoutException
import ru.tinkoff.acquiring.sdk.models.enums.DataTypeQr
import ru.tinkoff.acquiring.sdk.models.enums.ResponseStatus
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.payment.base.PaymentUiEvent
import ru.tinkoff.acquiring.sdk.payment.methods.InitConfigurator.configure
import ru.tinkoff.acquiring.sdk.payment.methods.requiredPaymentId
import ru.tinkoff.acquiring.sdk.payment.pooling.GetStatusMethod
import ru.tinkoff.acquiring.sdk.payment.pooling.GetStatusPooling
import ru.tinkoff.acquiring.sdk.redesign.sbp.util.BankInfo
import ru.tinkoff.acquiring.sdk.redesign.sbp.util.NspkBankAppsProvider
import ru.tinkoff.acquiring.sdk.redesign.sbp.util.NspkInstalledAppsChecker

/**
 * Created by i.golovachev
 */
class SbpPaymentProcess internal constructor(
    private val sdk: AcquiringSdk,
    private val bankAppsProvider: NspkInstalledAppsChecker,
    private val nspkBankProvider: NspkBankAppsProvider,
    private val getStatusPooling: GetStatusPooling,
    private val scope: CoroutineScope = CoroutineScope(Dispatchers.IO)
) {
    internal constructor(
        sdk: AcquiringSdk,
        bankAppsProvider: NspkInstalledAppsChecker,
        nspkBankAppsProvider: NspkBankAppsProvider,
        ioDispatcher: CoroutineDispatcher = Dispatchers.IO
    ) : this(
        sdk,
        bankAppsProvider,
        nspkBankAppsProvider,
        GetStatusPooling(
            GetStatusMethod.Impl(sdk)),
        CoroutineScope(ioDispatcher)
    )

    private val _stateFlow = MutableStateFlow<SbpPaymentState>(SbpPaymentState.Created)
    val stateFlow: StateFlow<SbpPaymentState> = _stateFlow.asStateFlow()
    private var looperJob: Job = Job()

    fun start(options: PaymentOptions) {
        scope.launch {
            runOrCatch {
                val nspkApps = nspkBankProvider.getNspkApps()
                val paymentId = options.paymentId ?: sendInit(options)
                setState(SbpPaymentState.Started(paymentId))
                val deeplink = sendGetQr(paymentId)
                val installedApps = bankAppsProvider.checkInstalledApps(nspkApps, deeplink)
                setState(SbpPaymentState.NeedChooseOnUi(paymentId, PaymentUiEvent.ShowApps(installedApps)))
            }
        }
    }

    fun goingToBankApp(bankInfo: BankInfo) {
        when (val currentState = _stateFlow.value) {
            is SbpPaymentState.NeedChooseOnUi -> {
                setState(SbpPaymentState.LeaveOnBankApp(
                    paymentId = currentState.paymentId,
                    bankInfo = bankInfo,
                ))
            }
            is SbpPaymentState.Stopped -> {
                setState(SbpPaymentState.LeaveOnBankApp(
                    paymentId = currentState.paymentId!!,
                    bankInfo = bankInfo,
                ))
            }
            else -> Unit
        }
    }

    fun startCheckingStatus(retriesCount: Int? = null) {
        val currentState = _stateFlow.value
        val paymentId = currentState.paymentId ?: return
        if (currentState is SbpPaymentState.LeaveOnBankApp || currentState is SbpPaymentState.CheckingPaused) {
            setState(SbpPaymentState.CheckingStatus(paymentId, null))
            looperJob.cancel()
            looperJob = startLoping(retriesCount, paymentId = paymentId)
        }
    }

    fun pause() {
        val currentState = _stateFlow.value
        if (currentState is SbpPaymentState.CheckingStatus) {
            setState(SbpPaymentState.CheckingPaused(currentState.paymentId))
            if (looperJob.isActive) {
                looperJob.cancel()
            }
        }
    }

    fun stop() {
        setState(SbpPaymentState.Stopped(_stateFlow.value.paymentId))
        if (looperJob.isActive) {
            looperJob.cancel()
        }
    }

    private fun setState(newState: SbpPaymentState) {
        _stateFlow.value = newState
    }

    private suspend fun runOrCatch(block: suspend () -> Unit) = try {
        block()
    } catch (throwable: Throwable) {
        _stateFlow.update {
            if (throwable is CancellationException) {
                SbpPaymentState.Stopped(it.paymentId)
            } else {
                SbpPaymentState.GetBankListFailed(it.paymentId, throwable)
            }
        }
    }

    private suspend fun sendInit(paymentOptions: PaymentOptions): Long {
        val response = sdk.init { configure(paymentOptions) }.execute()
        return response.requiredPaymentId()
    }

    private suspend fun sendGetQr(paymentId: Long): String = checkNotNull(
        sdk.getQr {
            this.paymentId = paymentId
            this.dataType = DataTypeQr.PAYLOAD
        }.execute().data,
    ) { "data from NSPK are null" }

    private fun startLoping(retriesCount: Int?, paymentId: Long): Job {
        return scope.launch {
            getStatusPooling.start(retriesCount = retriesCount, paymentId = paymentId)
                .map {
                    SbpPaymentState.mapResponseStatusToState(
                        status = it,
                        paymentId = paymentId
                    )
                }.catch {
                    emit(SbpPaymentState.PaymentFailed(throwable = it, paymentId = paymentId))
                }
                .collectLatest {
                    setState(it)
                }
        }
    }
}

sealed interface SbpPaymentState {
    val paymentId: Long?

    object Created : SbpPaymentState {
        override val paymentId: Long? = null
    }

    class Started(override val paymentId: Long) : SbpPaymentState

    class NeedChooseOnUi(
        override val paymentId: Long,
        val showApps: PaymentUiEvent.ShowApps
    ) : SbpPaymentState

    class GetBankListFailed(override val paymentId: Long?, val throwable: Throwable) :
        SbpPaymentState

    class LeaveOnBankApp(
        override val paymentId: Long,
        val bankInfo: BankInfo,
    ) : SbpPaymentState

    class CheckingStatus(
        override val paymentId: Long,
        val status: ResponseStatus?
    ) : SbpPaymentState

    class CheckingPaused(
        override val paymentId: Long
    ) : SbpPaymentState

    class PaymentFailed(override val paymentId: Long?, val throwable: Throwable) : SbpPaymentState

    class Success(override val paymentId: Long, val cardId: String?, val rebillId: String?) :
        SbpPaymentState

    class Stopped(override val paymentId: Long?) : SbpPaymentState

    companion object {

        fun mapResponseStatusToState(status: ResponseStatus, paymentId: Long) = when (status) {
            in ResponseStatus.successStatuses -> {
                Success(
                    paymentId, null, null
                )
            }
            in ResponseStatus.unhappyPassStatuses -> {
                PaymentFailed(
                    paymentId,
                    AcquiringSdkException(IllegalStateException("PaymentState = $status"))
                )
            }
            ResponseStatus.DEADLINE_EXPIRED -> {
                PaymentFailed(
                    paymentId,
                    AcquiringSdkTimeoutException(IllegalStateException("PaymentState = $status"))
                )
            }
            else -> CheckingStatus(paymentId, status)
        }
    }
}
