package ru.tinkoff.acquiring.sdk.payment

import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancelChildren
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkTimeoutException
import ru.tinkoff.acquiring.sdk.exceptions.getErrorCodeIfApiError
import ru.tinkoff.acquiring.sdk.models.enums.ResponseStatus
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.payment.methods.TpayMethods
import ru.tinkoff.acquiring.sdk.payment.pooling.GetStatusPooling

/**
 * Created by i.golovachev
 */
class TpayProcess internal constructor(
    private val getStatusPooling: GetStatusPooling,
    private val getTpayLinkMethods: TpayMethods,
    private val scope: CoroutineScope
) {
    private val _stateFlow = MutableStateFlow<TpayPaymentState>(TpayPaymentState.Created)
    val stateFlow = _stateFlow.asStateFlow()
    private var looperJob: Job = Job()

    fun start(
        paymentOptions: PaymentOptions,
        tpayVersion: String,
    ) {
        scope.launch {
            try {
                startFlow(paymentOptions, tpayVersion)
            } catch (ignored: CancellationException) {
                throw ignored
            } catch (e: Exception) {
                setState(TpayPaymentState.PaymentFailed(_stateFlow.value.paymentId, e, e.getErrorCodeIfApiError()))
            }
        }
    }

    fun goingToBankApp() {
        when (val currentState = _stateFlow.value) {
            is TpayPaymentState.NeedChooseOnUi -> {
                setState(TpayPaymentState.LeaveOnBankApp(currentState.paymentId!!))
            }
            else -> Unit
        }
    }

    fun startCheckingStatus() {
        val currentState = _stateFlow.value
        val paymentId = currentState.paymentId ?: return
        if (currentState is TpayPaymentState.LeaveOnBankApp || currentState is TpayPaymentState.Paused) {
            setState(TpayPaymentState.CheckingStatus(paymentId))
            looperJob.cancel()
            looperJob = startCheckStatusLooping(paymentId)
        }
    }

    fun pause() {
        val currentState = _stateFlow.value
        if (currentState is TpayPaymentState.CheckingStatus) {
            setState(TpayPaymentState.Paused(currentState.paymentId))
            if (looperJob.isActive) {
                looperJob.cancel()
            }
        }
    }

    fun stop() {
        setState(TpayPaymentState.Stopped(_stateFlow.value.paymentId))
        if (scope.isActive) {
            scope.coroutineContext.cancelChildren()
        }
    }
    
    private fun setState(newState: TpayPaymentState) {
        _stateFlow.value = newState
    }

    private suspend fun startFlow(
        paymentOptions: PaymentOptions,
        tpayVersion: String,
    ) {
        val paymentId = paymentOptions.paymentId ?: init(paymentOptions)
        setState(TpayPaymentState.Started(paymentId))
        val link = getLink(paymentId, tpayVersion)
        setState(TpayPaymentState.NeedChooseOnUi(paymentId, link))
    }

    private suspend fun init(paymentOptions: PaymentOptions): Long {
        return getTpayLinkMethods.init(paymentOptions = paymentOptions)
    }

    private suspend fun getLink(paymentId: Long, tpayVersion: String): String {
        return getTpayLinkMethods.tinkoffPayLink(paymentId, tpayVersion)
    }

    private fun startCheckStatusLooping(paymentId: Long): Job {
        return scope.launch {
            getStatusPooling.start(paymentId = paymentId)
                .map {
                    TpayPaymentState.mapResponseStatusToState(
                        status = it,
                        paymentId = paymentId
                    )
                }
                .catch {
                    emit(TpayPaymentState.PaymentFailed(throwable = it, paymentId = paymentId))
                }
                .collectLatest {
                    setState(it)
                }
        }
    }
}

sealed interface TpayPaymentState {
    val paymentId: Long?

    object Created : TpayPaymentState {
        override val paymentId: Long? = null
    }

    class Started(override val paymentId: Long) : TpayPaymentState

    class NeedChooseOnUi(
        override val paymentId: Long,
        val deeplink: String
    ) : TpayPaymentState

    class LeaveOnBankApp(override val paymentId: Long) : TpayPaymentState

    class CheckingStatus(
        override val paymentId: Long,
        val status: ResponseStatus? = null
    ) : TpayPaymentState

    class Paused(
        override val paymentId: Long,
    ) : TpayPaymentState

    class PaymentFailed(
        override val paymentId: Long?,
        val throwable: Throwable,
        val errorCode: String? = null
    ) : TpayPaymentState

    class Success(override val paymentId: Long, val cardId: String?, val rebillId: String?) :
        TpayPaymentState

    class Stopped(override val paymentId: Long?) : TpayPaymentState

    companion object {

        fun mapResponseStatusToState(status: ResponseStatus, paymentId: Long) = when (status) {
            ResponseStatus.AUTHORIZED, ResponseStatus.CONFIRMED, ResponseStatus.CONFIRMING -> {
                Success(
                    paymentId, null, null
                )
            }
            // по идее, в эти статусы проверка не зайдет - они обрабатываются в getStatusPooling
            ResponseStatus.REJECTED -> {
                PaymentFailed(
                    paymentId,
                    AcquiringSdkException(IllegalStateException("PaymentState = $status"))
                )
            }
            ResponseStatus.DEADLINE_EXPIRED -> {
                PaymentFailed(
                    paymentId,
                    AcquiringSdkTimeoutException(IllegalStateException("PaymentState = $status"))
                )
            }
            else -> CheckingStatus(paymentId, status)
        }
    }
}
