package ru.tinkoff.acquiring.sdk.payment.methods

import android.content.Context
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.paysources.CardSource
import ru.tinkoff.acquiring.sdk.responses.Check3dsVersionResponse
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsDataCollector
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.utils.CoroutineManager

/**
 * Created by i.golovachev
 */
interface Check3DsVersionMethods {

    class Data(
        val threeDsVersion: String,
        val additionalData: Map<String, String>,
        val paymentSystem: String?,
        val dsResponse: Check3dsVersionResponse
    )

    suspend fun callCheck3DsVersion(
        paymentId: Long,
        paymentSource: CardSource,
        paymentOptions: PaymentOptions,
        email: String? = null
    ): Data

}

internal class Check3DsVersionMethodsSdkImpl(
    private val sdk: AcquiringSdk,
    private val application: Context,
    private val threeDsDataCollector: ThreeDsDataCollector,
    private val featureToggleManager: FeatureToggleManager,
    private val coroutineManager: CoroutineManager = CoroutineManager(),
    private val default3dsVersion: String = "2",
) : Check3DsVersionMethods {
    override suspend fun callCheck3DsVersion(
        paymentId: Long,
        paymentSource: CardSource,
        paymentOptions: PaymentOptions,
        email: String?
    ): Check3DsVersionMethods.Data {
        val check3Ds = check3Ds(paymentId, paymentSource).execute()
        add3dsVersionToContext(check3Ds)
        val data = getAdditionalData(check3Ds)
        val check3DsVersion = check3Ds.version ?: default3dsVersion
        return Check3DsVersionMethods.Data(
            check3DsVersion, data, check3Ds.paymentSystem, check3Ds
        )
    }

    private fun add3dsVersionToContext(result: Check3dsVersionResponse) {
        try {
            val threeDsVersion = when {
                result.isAppBase() -> FeatureToggleManager.FeatureToggleContext.VERSION_2_APP
                result.is3DsVersionV2() -> FeatureToggleManager.FeatureToggleContext.VERSION_2_BROWSER
                else -> FeatureToggleManager.FeatureToggleContext.VERSION_1
            }
            coroutineManager.launchOnBackground {
                featureToggleManager
                    .updateFeatureToggleContext {
                        it.copy(
                            threeDsVersion = threeDsVersion,
                            appVersion = AcquiringSdk.appVersion,
                        )
                    }
            }
        } catch (e: Exception) {
            AcquiringSdk.log(e)
        }
    }

    private fun check3Ds(paymentId: Long, paymentSource: CardSource) = sdk.check3DsVersion {
        this.paymentId = paymentId
        this.paymentSource = paymentSource
    }

    private suspend fun getAdditionalData(check3Ds: Check3dsVersionResponse): MutableMap<String, String> {
        val data = mutableMapOf<String, String>()
        if (check3Ds.serverTransId != null) {
            coroutineManager.withMain {
                data.putAll(threeDsDataCollector(application, check3Ds))
            }
        }
        return data
    }
}
