package ru.tinkoff.acquiring.sdk.redesign.cards.list.presentation

import androidx.annotation.VisibleForTesting
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringNetworkError
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringNetworkException
import ru.tinkoff.acquiring.sdk.exceptions.checkCustomerNotFoundError
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.enums.CardStatus
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.redesign.cards.list.models.CardItemUiModel
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.CardListCommand
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.CardListMode
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.CardsListFragmentArgs
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.CardsListState
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.MenuMode
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.ScreenMode
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.ScreenState
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.responses.GetCardListResponse
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.utils.BankCaptionProvider
import ru.tinkoff.acquiring.sdk.utils.ConnectionChecker
import ru.tinkoff.acquiring.sdk.utils.CoroutineManager
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory
import ru.tinkoff.acquiring.sdk.utils.checkNotNull
import ru.tinkoff.acquiring.sdk.utils.panSuffix

internal class CardsListFragmentViewModel(
    savedStateHandle: SavedStateHandle,
    private val sdk: AcquiringSdk,
    private val connectionChecker: ConnectionChecker,
    private val bankCaptionProvider: BankCaptionProvider,
    private val coroutineManager: CoroutineManager,
) : ViewModel() {

    private val args = CardsListFragmentArgs.fromSavedStateHandle(savedStateHandle)

    private val savedCardsOptions = args.savedCardsOptions
        .checkNotNull { "Не передан SavedCardsOptions" }

    private val selectedCardIdFlow = MutableStateFlow(args.card?.cardId ?: savedCardsOptions.features.selectedCardId)

    private var deleteJob: Job? = null

    @VisibleForTesting
    val stateFlow = MutableStateFlow(ScreenState.DEFAULT.copy(
        allowNewCard = savedCardsOptions.allowNewCard,
        withArrowBack = savedCardsOptions.withArrowBack,
        screenMode = when(savedCardsOptions.mode) {
            SavedCardsOptions.Mode.LIST -> ScreenMode.LIST
            SavedCardsOptions.Mode.PAYMENT -> ScreenMode.PAYMENT
        }
    ))

    val stateUiFlow = stateFlow.asStateFlow()

    private val _commandFlow = MutableSharedFlow<CardListCommand>()
    val commandFlow = _commandFlow.asSharedFlow()

    private fun loadData() {
        val customerKey = savedCardsOptions.customer.customerKey
        if (connectionChecker.isOnline().not()) {
            stateFlow.update {
                it.copy(
                    listState = CardsListState.NoNetwork(
                        notification = NotificationFactory.getOfflineNotification {
                            ::onReloadButtonClick
                        }
                    ),
                    menuMode = MenuMode.EMPTY
                )
            }
            return
        }
        stateFlow.update {
            it.copy(
                listState = CardsListState.Shimmer,
                menuMode = MenuMode.EMPTY
            )
        }
        viewModelScope.launch {
            if (customerKey == null) {
                stateFlow.update {
                    val exception = IllegalStateException("customerKey не должен быть пустым")
                    it.copy(
                        listState = CardsListState.Error(
                            notification = NotificationFactory
                                .getErrorCommonNotification(exception) {{ onErrorButtonClick(exception) }}
                        ),
                        menuMode = MenuMode.EMPTY
                    )
                }
                return@launch
            }

            withContext(coroutineManager.io) {
                sdk.getCardList { this.customerKey = customerKey }
                    .executeFlow()
                    .collect { r ->
                        r.process(
                            onSuccess = { handleGetCardListResponse(it) },
                            onFailure = ::handleGetCardListError
                        )
                    }
            }
        }
    }

    fun deleteCard(model: CardItemUiModel) {
        if (deleteJob?.isActive == true) return

        deleteJob = viewModelScope.launch {
            _commandFlow.emit(CardListCommand.RemoveCardProgress(model))

            val customerKey = savedCardsOptions.customer.customerKey

            try {
                withContext(coroutineManager.io) {
                    sdk.removeCard {
                        this.cardId = model.id
                        this.customerKey = customerKey
                    }.execute()
                }
                handleDeleteCardSuccess(model)
            } catch (e: Exception) {
                _commandFlow.emit(
                    CardListCommand.ShowCardDeleteError(
                        error = e,
                        maskedPan = model.pan?.panSuffix()
                    )
                )
            }
        }
    }

    fun returnToBaseMode() {
        val baseMode = resolveBaseCardListMode()
        updateContentState(baseMode)
    }

    fun switchToDeleteMode() {
        updateContentState(CardListMode.DELETE)
    }

    fun chooseCard(model: CardItemUiModel) {
        if (stateFlow.value.screenMode == ScreenMode.PAYMENT) {
            viewModelScope.launch {
                _commandFlow.emit(CardListCommand.FinishWithSelectCard(model.card))
            }
        }
    }

    fun onPayByAnotherCardClicked() {
        payByNewCard()
    }

    fun onAddNewCardClicked() = viewModelScope.launch {
        goToAttachCard()
    }

    private fun onErrorButtonClick(error: Throwable) {
        viewModelScope.launch {
            _commandFlow.emit(CardListCommand.FinishWithError(error))
        }
    }

    private fun resolveBaseCardListMode(): CardListMode {
        return if (selectedCardIdFlow.value == null) CardListMode.LIST else CardListMode.CHOOSE
    }

    private fun updateContentState(mode: CardListMode) {
        stateFlow.update { state ->
            val prevListState = state.listState
            if (prevListState is CardsListState.Content) {
                val cards = prevListState.cards.map {
                    it.copy(
                        showDelete = mode == CardListMode.DELETE,
                        isBlocked = it.isBlocked,
                        showChoose = selectedCardIdFlow.value == it.card.cardId && mode === CardListMode.CHOOSE
                    )
                }
                state.copy(
                    listMode = mode,
                    allowNewCard = resolveAllowNewCard(mode),
                    menuMode = resolveMenuMode(mode, cards),
                    listState = CardsListState.Content(cards)
                )
            } else {
                state.copy(
                    listMode = mode,
                    allowNewCard = resolveAllowNewCard(mode),
                    menuMode = resolveMenuMode(mode)
                )
            }
        }
    }

    private suspend fun goToAttachCard() {
        val attachCardEvent = CardListCommand.ToAttachCard
        _commandFlow.emit(attachCardEvent)
    }

    fun onBackPressed() {
        viewModelScope.launch {
            when (stateFlow.value.listState) {
                is CardsListState.Content -> when (stateFlow.value.screenMode) {
                    ScreenMode.LIST -> {
                        _commandFlow.emit(CardListCommand.FinishWithCancel)
                    }

                    ScreenMode.PAYMENT -> {
                        val selectedCard =
                            (stateFlow.value.listState as? CardsListState.Content)?.cards
                                ?.find { it.id == selectedCardIdFlow.value }
                        if (selectedCard != null) {
                            _commandFlow.emit(
                                CardListCommand.FinishWithSelectCard(selectedCard.card)
                            )
                        } else {
                            _commandFlow.emit(CardListCommand.FinishWithCancel)
                        }
                    }
                }

                else -> {
                    _commandFlow.emit(CardListCommand.FinishWithCancel)
                }
            }
        }
    }

    private fun handleGetCardListResponse(it: GetCardListResponse) {
        try {
            val mode = if (selectedCardIdFlow.value != null) {
                CardListMode.CHOOSE
            } else {
                CardListMode.LIST
            }
            val uiCards = filterCards(it.cards, mode, savedCardsOptions.showOnlyRecurrentCards)
            stateFlow.update {
                it.copy(
                    listState = if (uiCards.isEmpty()) {
                        createEmptyListState()
                    } else {
                        CardsListState.Content(uiCards)
                    },
                    menuMode = resolveMenuMode(mode, uiCards)
                )
            }
        } catch (e: Exception) {
            handleGetCardListError(e)
        }
    }

    private fun filterCards(
        cards: Array<Card>,
        mode: CardListMode,
        showOnlyRecurrentCards: Boolean
    ): List<CardItemUiModel> {
        return cards
                .filter { card -> card.status == CardStatus.ACTIVE }
                .filter { card ->
                    if (showOnlyRecurrentCards) card.rebillId.isNullOrEmpty().not() else true
                }
            .map {
                val cardNumber = checkNotNull(it.pan)
                CardItemUiModel(
                    card = it,
                    bankName = bankCaptionProvider(cardNumber),
                    showChoose = (selectedCardIdFlow.value == it.cardId) && mode === CardListMode.CHOOSE
                )
            }
    }

    private fun handleGetCardListError(exception: Exception) {
        stateFlow.update {
            it.copy(
                listState = if (exception.checkCustomerNotFoundError()) {
                    createEmptyListState()
                } else {
                    CardsListState.Error(
                        notification = NotificationFactory
                            .getContentLoadingErrorNotification(exception){ e ->
                                when(e) {
                                    is AcquiringNetworkError -> {{ onErrorButtonClick(e) }}
                                    is AcquiringNetworkException -> ::loadData
                                    else -> {{ onErrorButtonClick(e) }}
                                }
                            }
                    )
                },
                menuMode = MenuMode.EMPTY
            )
        }
    }

    private fun handleDeleteCardSuccess(deletedCard: CardItemUiModel) {
        val currentListState = stateFlow.value.listState
        if (currentListState is CardsListState.Content) {
            val list = currentListState.cards.toMutableList()
            val indexAt = list.indexOfFirst { it.id == deletedCard.id }
            if (indexAt != -1) {
                list.removeAt(indexAt)
            }

            if (list.isEmpty()) {
                returnToBaseMode()
                stateFlow.update {
                    it.copy(
                        listState = createEmptyListState(),
                        menuMode = resolveMenuMode(it.listMode, null)
                    )
                }
            } else {
                if (deletedCard.showChoose || deletedCard.id == selectedCardIdFlow.value) {
                    selectedCardIdFlow.value = list.firstOrNull()?.id
                }
                stateFlow.update {
                    it.copy(
                        listState = CardsListState.Content(list),
                        menuMode = resolveMenuMode(it.listMode, list)
                    )
                }
            }
        }

        viewModelScope.launch {
            _commandFlow.emit(CardListCommand.RemoveCardSuccess(deletedCard))
        }
    }

    private fun createEmptyListState(): CardsListState.Empty {
        val subTitleTextRes: Int
        val buttonTextRes: Int
        val onButtonClick: (()->Unit)?
        if (stateFlow.value.screenMode == ScreenMode.PAYMENT) {
            subTitleTextRes = R.string.acq_cardlist_pay_description
            buttonTextRes = R.string.acq_cardlist_pay_button_add
            onButtonClick = {
                onPayByAnotherCardClicked()
            }
        } else {
            subTitleTextRes = R.string.acq_cardlist_description
            buttonTextRes = R.string.acq_cardlist_button_add
            onButtonClick = {
                onAddNewCardClicked()
            }
        }
        return CardsListState.Empty(
            notification = Notification(
                type = StatusViewData.Type.CARDS_EMPTY,
                description = subTitleTextRes,
                button = buttonTextRes.takeIf { stateFlow.value.allowNewCard },
                onClick = onButtonClick
            )
        )
    }

    private fun payByNewCard() {
        viewModelScope.launch {
            _commandFlow.emit(CardListCommand.ToPayByNewCard)
        }
    }

    private fun resolveMenuMode(mode: CardListMode, cards: List<CardItemUiModel>? = null): MenuMode {
        return when {
            cards.isNullOrEmpty() -> MenuMode.EMPTY
            (mode == CardListMode.LIST || mode == CardListMode.CHOOSE) -> MenuMode.EDIT
            mode == CardListMode.DELETE -> MenuMode.SUCCESS
            else -> MenuMode.EMPTY
        }
    }

    private fun resolveAllowNewCard(mode: CardListMode): Boolean {
        return when (mode) {
            CardListMode.DELETE -> false
            else -> savedCardsOptions.allowNewCard
        }
    }

    fun onReloadButtonClick() {
        loadData()
    }

    fun onRequestRefresh() {
        loadData()
    }
}
