package ru.tinkoff.acquiring.sdk.redesign.cards.list.ui

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.annotation.StringRes
import androidx.core.os.bundleOf
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.setFragmentResult
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.NavHostFragment
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.CardListGraphDirections
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqCardListFlowFragmentBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.AttachCardOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.CardListFlowViewModel.CardListFlowCommand
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.PaymentByCardFlowFragment
import ru.tinkoff.acquiring.sdk.ui.fragments.AttachCardFlowFragment
import ru.tinkoff.acquiring.sdk.utils.AcqSnackBarHelper

class CardListFlowFragment : Fragment(), IsolatedKoinComponent {

    private var viewBinding: AcqCardListFlowFragmentBinding? = null
    private val viewModel by viewModel<CardListFlowViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }
    private var navHostFragment: NavHostFragment? = null
    private var snackBarHelper: AcqSnackBarHelper? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val binding = AcqCardListFlowFragmentBinding.inflate(inflater, container, false)
        viewBinding = binding
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        this.snackBarHelper = AcqSnackBarHelper(view)

        val hostFragment =
            childFragmentManager.findFragmentById(R.id.main_content) as NavHostFragment
        navHostFragment = hostFragment

        CardsListFragment.registerResultListener(
            hostFragment.childFragmentManager,
            this
        ) { result ->
            when (result) {
                CardsListFragment.Result.Cancelled -> returnResult(Result.Cancel)
                is CardsListFragment.Result.CardSelected -> finishWithCard(
                    card = result.card
                )

                is CardsListFragment.Result.Error -> returnResult(
                    Result.Failed(
                        error = result.error,
                        errorCode = result.errorCode
                    )
                )

                CardsListFragment.Result.AddCard -> viewModel.onAttachCard()
                CardsListFragment.Result.PayByNewCard -> viewModel.onPayByNewCard()
                null -> Unit
            }
        }

        AttachCardFlowFragment.registerResultListener(hostFragment.childFragmentManager, this) {
            when (it) {
                is AttachCardFlowFragment.Result.Failed -> viewModel.onAttachCardError(
                    error = it.error,
                    panSuffix = it.panSuffix
                )

                is AttachCardFlowFragment.Result.Success -> viewModel.onAttachCardSuccess(
                    cardId = it.cardId,
                    panSuffix = it.panSuffix
                )

                AttachCardFlowFragment.Result.Cancel -> {
                    viewModel.handleEvent(CardListFlowViewModel.Event.AttachCardCancel)
                }

                null -> Unit
            }
        }

        PaymentByCardFlowFragment.registerResultListener(
            hostFragment.childFragmentManager,
            this
        ) { result ->
            when (result) {
                PaymentByCardFlowFragment.Result.PaymentCancel -> onPayByNewCardCancel()
                is PaymentByCardFlowFragment.Result.PaymentError -> onPayByNewCardError(
                    error = result.error,
                    paymentId = result.paymentId
                )

                is PaymentByCardFlowFragment.Result.PaymentSuccess -> onPayByNewCardSuccess(
                    paymentId = result.paymentId,
                    cardId = result.cardId,
                    rebillId = result.rebillId
                )
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                handleCommands()
            }
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        snackBarHelper = null
        viewBinding = null
    }

    private fun onPayByNewCardCancel() {
        viewModel.handleEvent(CardListFlowViewModel.Event.PayByNewCardCancel)
    }

    private suspend fun handleCommands() {
        viewModel.commandFlow.collect {
            when (it) {
                is CardListFlowCommand.OpenAttachCard -> startAttachCard(
                    it.attachCardOptions,
                    it.showError,
                )

                is CardListFlowCommand.OpenPayByNewCard -> payByNewCard(
                    it.paymentOptions,
                    it.isNewCardViewToggleEnabled
                )

                is CardListFlowCommand.ShowCardAddSuccess -> showAddCardSuccess(
                    it.panSuffix
                )

                is CardListFlowCommand.OpenCardsList -> {
                    showCardListScreen(it.savedCardsOptions, it.card)
                }

                is CardListFlowCommand.ReturnToCardList -> returnToCardList()
                is CardListFlowCommand.ShowCardAddError -> showAddCardError(
                    message = it.message,
                    args = it.args
                )
            }
        }
    }

    private fun showAddCardError(@StringRes message: Int, args: List<Any>) {
        snackBarHelper?.showWithIcon(
            R.drawable.acq_ic_alert_ngative,
            if (args.isNotEmpty()) {
                getString(message, *args.toTypedArray())
            } else {
                getString(message)
            }
        )
    }

    private fun returnToCardList() {
        navHostFragment?.navController?.popBackStack(R.id.cardListFragment, false)
    }

    private fun showAddCardSuccess(panSuffix: String) {
        snackBarHelper?.showWithIcon(
            R.drawable.acq_ic_card_sparkle,
            getString(R.string.acq_cardlist_snackbar_add, panSuffix)
        )
    }

    private fun showCardListScreen(
        savedCardsOptions: SavedCardsOptions,
        card: Card?
    ) {
        navHostFragment?.navController?.navigate(
            CardListGraphDirections.openCardList(
                savedCardsOptions = savedCardsOptions,
                card = card
            )
        )
    }

    private fun payByNewCard(paymentOptions: PaymentOptions, isNewCardViewToggleEnabled: Boolean) {
        navHostFragment?.navController?.navigate(
            CardListGraphDirections.payByNewCard(
                paymentOptions = paymentOptions,
                card = null,
                showArrow = true,
                isNewCardViewToggleEnabled = isNewCardViewToggleEnabled
            )
        )
    }

    private fun startAttachCard(attachCardOptions: AttachCardOptions, showError: Boolean) {
        navHostFragment?.navController?.navigate(
            CardListGraphDirections.openAttachCardFlow(
                attachCardOptions = attachCardOptions,
                showError = showError,
            )
        )
    }

    private fun finishWithCard(card: Card) {
        returnResult(Result.CardChosen(card))
    }

    private fun onPayByNewCardSuccess(
        paymentId: Long,
        cardId: String?,
        rebillId: String?
    ) {
        returnResult(
            Result.PaidByNewCard(
                paymentId = paymentId,
                cardId = cardId,
                rebillId = rebillId
            )
        )
    }

    private fun onPayByNewCardError(error: Throwable, paymentId: Long?) {
        returnResult(
            Result.Failed(
                error = error,
                paymentId = paymentId
            )
        )
    }

    private fun returnResult(result: Result) {
        setFragmentResult(
            RESULT_KEY, bundleOf(
                RESULT_DATA_KEY to result
            )
        )
    }

    companion object {
        private const val RESULT_KEY = "CardListFlowFragment.RESULT_KEY"
        private const val RESULT_DATA_KEY = "CardListFlowFragment.RESULT_DATA_KEY"

        fun newInstance(
            savedCardsOptions: SavedCardsOptions,
            paymentOptions: PaymentOptions? = null
        ): CardListFlowFragment {
            return CardListFlowFragment().apply {
                arguments = CardListFlowFragmentArgs(
                    savedCardsOptions = savedCardsOptions,
                    paymentOptions = paymentOptions,
                ).toBundle()
            }
        }

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(RESULT_KEY, lifecycleOwner) { _, bundle ->
                val result = bundle.getParcelable<Result>(RESULT_DATA_KEY)
                result?.let(listener)
            }
        }

    }

    sealed interface Result : Parcelable {
        @Parcelize
        data class CardChosen(
            val card: Card
        ) : Result

        @Parcelize
        data class Failed(
            val error: Throwable,
            val paymentId: Long? = null,
            val errorCode: Int? = null
        ) : Result

        @Parcelize
        data class PaidByNewCard(
            val paymentId: Long,
            val cardId: String?,
            val rebillId: String?
        ) : Result

        @Parcelize
        data object Cancel : Result
    }
}
