package ru.tinkoff.acquiring.sdk.redesign.cards.list.ui

import androidx.annotation.StringRes
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringApiException
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.AttachCardOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.network.AcquiringApi.API_ERROR_CODE_CARD_ALREADY_ATTACHED
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.toggles.toggles.PaymentByCardV2FeatureToggle
import ru.tinkoff.acquiring.sdk.utils.panSuffix

/**
 * @author s.y.biryukov
 */
internal class CardListFlowViewModel(
    stateHandle: SavedStateHandle,
    private val featureToggleManager: FeatureToggleManager,
) : ViewModel() {

    private val args = CardListFlowFragmentArgs.fromSavedStateHandle(stateHandle)

    private val savedCardsOptions = args.savedCardsOptions
    private val paymentOptions = args.paymentOptions

    private val _commandFlow = MutableSharedFlow<CardListFlowCommand>()
    val commandFlow = _commandFlow.asSharedFlow()

    private var isNewCardViewToggleEnabled: Boolean = false

    init {
        viewModelScope.launch {
            _commandFlow.emit(
                CardListFlowCommand.OpenCardsList(
                    savedCardsOptions = savedCardsOptions,
                    card = args.card
                )
            )
            isNewCardViewToggleEnabled = featureToggleManager.isEnabled(PaymentByCardV2FeatureToggle)
        }
    }

    fun handleEvent(event: Event) {
        when (event) {
            Event.PayByNewCardCancel -> onPayByNewCardCancel()
            Event.AttachCardCancel -> onAttachCardCancel()
        }
    }

    fun onPayByNewCard() {
        val options = requireNotNull(paymentOptions) { "Не задан параметр PaymentOptions" }

        viewModelScope.launch {
            _commandFlow.emit(
                CardListFlowCommand.OpenPayByNewCard(
                    paymentOptions = options,
                    isNewCardViewToggleEnabled = isNewCardViewToggleEnabled
                )
            )
        }
    }

    fun onAttachCard() {
        val attachCardOptions = AttachCardOptions().setOptions {
            setTerminalParams(savedCardsOptions.terminalKey, savedCardsOptions.publicKey)
            customerOptions {
                checkType = savedCardsOptions.customer.checkType
                customerKey = savedCardsOptions.customer.customerKey
            }
            features = savedCardsOptions.features
        }

        viewModelScope.launch {
            _commandFlow.emit(
                CardListFlowCommand.OpenAttachCard(
                    attachCardOptions = attachCardOptions,
                    showError = false,
                )
            )
        }
    }

    fun onAttachCardSuccess(
        cardId: String,
        panSuffix: String
    ) {
        viewModelScope.launch {
            _commandFlow.emit(
                CardListFlowCommand.ShowCardAddSuccess(
                    cardId = cardId,
                    panSuffix = panSuffix
                )
            )
            _commandFlow.emit(
                CardListFlowCommand.ReturnToCardList
            )
        }
    }

    fun onAttachCardError(error: Throwable, panSuffix: String?) {
        AcquiringSdk.log(error)
        var message = R.string.acq_cardlist_snackbar_add_error_nopan
        var args = emptyList<Any>()
        when {
            (error as? AcquiringApiException)?.errorCode == API_ERROR_CODE_CARD_ALREADY_ATTACHED -> {
                message = R.string.acq_cardlist_snackbar_add_error_already_attached
            }

            panSuffix != null -> {
                message = R.string.acq_cardlist_snackbar_add_error
                args = listOf(panSuffix.panSuffix())
            }
        }
        viewModelScope.launch {
            _commandFlow.emit(
                CardListFlowCommand.ShowCardAddError(
                    message = message,
                    args = args
                )
            )
        }
        returnToCardList()
    }

    private fun onAttachCardCancel() {
        returnToCardList()
    }

    private fun onPayByNewCardCancel() {
        returnToCardList()
    }

    private fun returnToCardList() {
        viewModelScope.launch {
            _commandFlow.emit(CardListFlowCommand.ReturnToCardList)
        }
    }

    sealed interface Event {
        data object PayByNewCardCancel : Event
        data object AttachCardCancel : Event
    }

    internal sealed interface CardListFlowCommand {
        data class OpenCardsList(
            val savedCardsOptions: SavedCardsOptions,
            val card: Card? = null
        ) : CardListFlowCommand

        data class OpenPayByNewCard(
            val paymentOptions: PaymentOptions,
            val isNewCardViewToggleEnabled: Boolean,
        ) : CardListFlowCommand

        data class OpenAttachCard(
            val attachCardOptions: AttachCardOptions,
            val showError: Boolean,
        ) : CardListFlowCommand

        data class ShowCardAddSuccess(
            val cardId: String,
            val panSuffix: String,
        ) : CardListFlowCommand

        data class ShowCardAddError(
            @StringRes val message: Int,
            val args: List<Any> = emptyList(),
        ) : CardListFlowCommand

        data object ReturnToCardList : CardListFlowCommand
    }
}
