package ru.tinkoff.acquiring.sdk.redesign.cards.list.ui

import android.content.Intent
import android.os.Bundle
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ChooseCardLauncher
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ChooseCardLauncher.Contract.EXTRA_PAYMENT_OPTIONS
import ru.tinkoff.acquiring.sdk.redesign.common.LauncherConstants
import ru.tinkoff.acquiring.sdk.redesign.common.LauncherConstants.EXTRA_CARD_ID
import ru.tinkoff.acquiring.sdk.redesign.common.LauncherConstants.EXTRA_ERROR
import ru.tinkoff.acquiring.sdk.redesign.common.LauncherConstants.EXTRA_PAYMENT_ID
import ru.tinkoff.acquiring.sdk.redesign.common.LauncherConstants.EXTRA_REBILL_ID
import ru.tinkoff.acquiring.sdk.redesign.common.LauncherConstants.RESULT_ERROR
import ru.tinkoff.acquiring.sdk.toggles.DeviceIdProvider
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.utils.getOptions
import ru.tinkoff.acquiring.sdk.utils.getParcelable

internal class CardsListActivity : AppCompatActivity(), IsolatedKoinComponent {

    private val featureToggleManager: FeatureToggleManager by inject()
    private val deviceIdProvider: DeviceIdProvider by inject()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.acq_activity_card_list)

        val savedCardsOptions: SavedCardsOptions = intent.getOptions()

        updateFeatureToggleContext(savedCardsOptions.terminalKey)

        CardListFlowFragment.registerResultListener(supportFragmentManager, this) {
            when (it) {
                CardListFlowFragment.Result.Cancel -> {
                    setResult(RESULT_CANCELED)
                    finish()
                }

                is CardListFlowFragment.Result.Failed -> {
                    val intent = Intent().apply {
                        putExtra(EXTRA_ERROR, it.error)
                        putExtra(EXTRA_PAYMENT_ID, it.paymentId)
                    }
                    setResult(RESULT_ERROR, intent)
                    finish()
                }

                is CardListFlowFragment.Result.CardChosen -> {
                    setResult(
                        RESULT_OK,
                        Intent().putExtra(
                            ChooseCardLauncher.Contract.EXTRA_CHOSEN_CARD,
                            it.card
                        )
                    )
                    finish()
                }

                is CardListFlowFragment.Result.PaidByNewCard -> {
                    val intent = Intent()
                    intent.putExtra(EXTRA_PAYMENT_ID, it.paymentId)
                    intent.putExtra(EXTRA_CARD_ID,  it.cardId)
                    intent.putExtra(EXTRA_REBILL_ID, it.rebillId)
                    setResult(
                        LauncherConstants.RESULT_PAID_BY_NEW_CARD,
                        intent
                    )
                    finish()
                }

                null -> Unit
            }
        }

        if (savedInstanceState == null) {
            val paymentOptions = intent.getParcelable(EXTRA_PAYMENT_OPTIONS, PaymentOptions::class)

            val flowFragment = CardListFlowFragment.newInstance(savedCardsOptions, paymentOptions)
            supportFragmentManager.beginTransaction()
                .add(R.id.acq_card_list_root, flowFragment)
                .commit()
        }
    }

    private fun updateFeatureToggleContext(terminalKey: String) {
        lifecycleScope.launch {
            featureToggleManager.updateFeatureToggleContext {
                it.copy(
                    terminalKey = terminalKey,
                    deviceId = deviceIdProvider.getDeviceId(),
                    appVersion = AcquiringSdk.appVersion,
                )
            }
        }
    }
}
