package ru.tinkoff.acquiring.sdk.redesign.cards.list.ui

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.widget.TextView
import androidx.activity.addCallback
import androidx.core.os.bundleOf
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.children
import androidx.core.view.isVisible
import androidx.core.view.updatePadding
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.recyclerview.widget.RecyclerView
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentCardListBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.redesign.cards.list.adapters.CardsListAdapter
import ru.tinkoff.acquiring.sdk.redesign.cards.list.presentation.CardsListFragmentViewModel
import ru.tinkoff.acquiring.sdk.redesign.common.util.AcqShimmerAnimator
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.utils.AcqSnackBarHelper
import ru.tinkoff.acquiring.sdk.utils.menuItemVisible
import ru.tinkoff.acquiring.sdk.utils.showById
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

class CardsListFragment : Fragment(), IsolatedKoinComponent {
    private var viewBinding: AcqFragmentCardListBinding? = null


    private val viewModel by viewModel<CardsListFragmentViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }

    private val recyclerView: RecyclerView?
        get() = requireView().findViewById(R.id.acq_card_list_view)

    private val cardShimmer: ViewGroup?
        get() = requireView().findViewById(R.id.acq_card_list_shimmer)

    private val addNewCard: TextView?
        get() = requireView().findViewById(R.id.acq_add_new_card)

    private val anotherCard: TextView?
        get() = requireView().findViewById(R.id.acq_another_card)

    private var cardsListAdapter: CardsListAdapter? = null

    private var snackBarHelper: AcqSnackBarHelper? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return AcqFragmentCardListBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        this.snackBarHelper = AcqSnackBarHelper(view)

        initToolbar()
        initViews()
        subscribeOnState()

        applyInsets(view)
    }

    private fun applyInsets(view: View) {
        ViewCompat.setOnApplyWindowInsetsListener(view) { v, insets ->
            v.updatePadding(
                bottom = insets.getInsets(WindowInsetsCompat.Type.navigationBars()).bottom
            )
            insets
        }
    }

    override fun onResume() {
        super.onResume()
        requestRefresh()

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.onBackPressed()
        }
    }

    private fun requestRefresh() {
        viewModel.onRequestRefresh()
    }

    private fun initToolbar() {
        val binding = viewBinding ?: return
        val toolbar = binding.acqToolbar
        toolbar.setNavigationOnClickListener {
            viewModel.onBackPressed()
        }
        toolbar.inflateMenu(R.menu.acq_card_list_menu)
        toolbar.setOnMenuItemClickListener {
            when (it.itemId) {
                R.id.acq_card_list_action_change -> {
                    viewModel.switchToDeleteMode()
                    true
                }

                R.id.acq_card_list_action_complete -> {
                    viewModel.returnToBaseMode()
                    true
                }

                else -> false
            }
        }
    }

    private fun initViews() {
        cardsListAdapter = CardsListAdapter(
            onDeleteClick = { viewModel.deleteCard(it) },
            onChooseClick = { viewModel.chooseCard(it) }
        )
        recyclerView?.adapter = cardsListAdapter
        addNewCard?.setOnClickListener {
            viewModel.onAddNewCardClicked()
        }
        anotherCard?.setOnClickListener {
            viewModel.onPayByAnotherCardClicked()
        }
    }

    private fun subscribeOnState() {
        lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                subscribeOnUiState()
                subscribeOnCommand()
            }
        }
    }

    private fun CoroutineScope.subscribeOnUiState() {
        launch {
            viewModel.stateUiFlow.collectLatest { state ->
                val binding = viewBinding ?: return@collectLatest
                val toolbar = binding.acqToolbar
                val mode = viewModel.stateUiFlow.value.menuMode
                val menu = toolbar.menu
                menu.menuItemVisible(R.id.acq_card_list_action_change, mode == MenuMode.EDIT)
                menu.menuItemVisible(R.id.acq_card_list_action_complete, mode == MenuMode.SUCCESS)

                if (state.screenMode == ScreenMode.PAYMENT) {
                    addNewCard?.isVisible = false
                    anotherCard?.isVisible = state.allowNewCard
                } else {
                    addNewCard?.isVisible = state.allowNewCard
                    anotherCard?.isVisible = false
                }

                if (state.withArrowBack) {
                    toolbar.setNavigationIcon(R.drawable.acq_arrow_back)
                }

                if (state.screenMode == ScreenMode.PAYMENT) {
                    toolbar.setTitle(R.string.acq_cardpay_title)
                } else {
                    toolbar.setTitle(R.string.acq_cardlist_title)
                }

                when (val listState = state.listState) {
                    is CardsListState.Content -> {
                        binding.acqViewFlipper.showById(R.id.acq_card_list_content)
                        cardsListAdapter?.setCards(listState.cards)
                    }

                    is CardsListState.Shimmer -> {
                        binding.acqViewFlipper.showById(R.id.acq_card_list_shimmer)
                        cardShimmer?.children?.toList()?.let {
                            AcqShimmerAnimator.animateSequentially(it)
                        }
                    }

                    is CardsListState.Error -> {
                        showStub(listState.notification)
                    }

                    is CardsListState.Empty -> {
                        showStub(listState.notification)
                    }

                    is CardsListState.NoNetwork -> {
                        showStub(listState.notification)
                    }
                }
            }
        }
    }

    private fun CoroutineScope.subscribeOnCommand() {
        launch {
            viewModel.commandFlow.collect {
                when (it) {
                    is CardListCommand.ToAttachCard -> setAttackCardResult()
                    is CardListCommand.ToPayByNewCard -> setPayByNewCardResult()
                    is CardListCommand.FinishWithError -> setErrorResult(it.throwable)
                    is CardListCommand.FinishWithSelectCard -> finishWithCard(it.selectedCard)
                    is CardListCommand.FinishWithCancel -> setCancelResult()
                    is CardListCommand.RemoveCardProgress -> showRemoveCardProgress(it.deletedCard.tail)
                    is CardListCommand.RemoveCardSuccess -> {
                        hideProgress()
                        snackBarHelper?.hide()
                    }
                    is CardListCommand.ShowCardDeleteError -> {
                        hideProgress()
                        snackBarHelper?.showWithIcon(
                            R.drawable.acq_ic_alert_ngative,
                            getString(R.string.acq_cardlist_snackbar_remove_error, it.maskedPan)
                        )
                    }
                }
            }
        }
    }

    private fun showStub(
        notification: Notification
    ) {
        viewBinding?.acqViewFlipper?.showById(R.id.acq_card_list_stub)
        viewBinding?.acqCardListStub?.fill(notification.toStatusViewData(requireContext()))
    }

    private fun showRemoveCardProgress(cardTail: String?) {
        requireActivity().window.setFlags(
            WindowManager.LayoutParams.FLAG_NOT_TOUCHABLE,
            WindowManager.LayoutParams.FLAG_NOT_TOUCHABLE
        )
        snackBarHelper?.showProgress(
            getString(
                R.string.acq_cardlist_snackbar_remove_progress,
                cardTail
            )
        )
    }

    private fun hideProgress() {
        requireActivity().window.clearFlags(WindowManager.LayoutParams.FLAG_NOT_TOUCHABLE)
    }

    private fun finishWithCard(card: Card) {
        setResult(Result.CardSelected(card))
    }

    private fun setPayByNewCardResult() {
        setResult(Result.PayByNewCard)
    }

    private fun setAttackCardResult() {
        setResult(Result.AddCard)
    }

    private fun setErrorResult(error: Throwable, errorCode: Int? = null) {
        setResult(
            Result.Error(
                error = error,
                errorCode = errorCode,
            )
        )
    }

    private fun setCancelResult() {
        setResult(Result.Cancelled)
    }

    override fun onDestroyView() {
        super.onDestroyView()
        recyclerView?.adapter = null
        cardsListAdapter = null
        viewBinding = null
        snackBarHelper = null
    }

    private fun setResult(addCard: Result) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to addCard
            )
        )
    }

    companion object {

        private const val FRAGMENT_RESULT_KEY = "CardListFragment.FRAGMENT_RESULT_KEY"
        private const val FRAGMENT_RESULT_BUNDLE_KEY = "CardListFragment.FRAGMENT_RESULT_BUNDLE_KEY"

        fun newInstance(
            savedCardsOptions: SavedCardsOptions,
            paymentOptions: PaymentOptions,
        ) : CardsListFragment {
            return CardsListFragment().apply {
                arguments = CardsListFragmentArgs(
                    savedCardsOptions = savedCardsOptions,
                    paymentOptions = paymentOptions
                ).toBundle()
            }
        }

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result?) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(FRAGMENT_RESULT_KEY, lifecycleOwner) { _, bundle ->
                listener.invoke(bundle.getParcelable(FRAGMENT_RESULT_BUNDLE_KEY))
            }
        }
    }


    sealed interface Result : Parcelable {
        @Parcelize
        data class CardSelected(val card: Card) : Result

        @Parcelize
        data object Cancelled : Result

        @Parcelize
        data class Error(
            val error: Throwable,
            val errorCode: Int? = null,
        ) : Result

        @Parcelize
        data object AddCard : Result

        @Parcelize
        data object PayByNewCard : Result

    }
}
