package ru.tinkoff.acquiring.sdk.redesign.common.carddatainput

import android.content.Context
import android.os.Bundle
import android.text.method.PasswordTransformationMethod
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.fragment.app.Fragment
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.cardscanners.delegate.CardScannerWrapper
import ru.tinkoff.acquiring.sdk.cardscanners.delegate.ScannedCardResult
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentCardDataInputBinding
import ru.tinkoff.acquiring.sdk.models.options.screen.BaseAcquiringOptions
import ru.tinkoff.acquiring.sdk.smartfield.BaubleCardLogo
import ru.tinkoff.acquiring.sdk.smartfield.BaubleClearButton
import ru.tinkoff.acquiring.sdk.smartfield.BaubleClearOrScanButton
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.CardPaymentSystem
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.validators.CardValidator
import ru.tinkoff.acquiring.sdk.utils.SimpleTextWatcher.Companion.afterTextChanged
import ru.tinkoff.decoro.MaskImpl
import ru.tinkoff.decoro.parser.UnderscoreDigitSlotsParser
import ru.tinkoff.decoro.watchers.MaskFormatWatcher
import java.lang.ref.WeakReference

internal class CardDataInputFragment : Fragment() {

    private val paymentOptions: BaseAcquiringOptions
        get() {
            return requireNotNull(arguments?.getParcelable(ARGUMENT_OPTIONS)) { "getParcelable(ARGUMENT_OPTIONS) is null" }
        }

    private var viewBinding: AcqFragmentCardDataInputBinding? = null
    var onComplete: ((CardDataInputFragment) -> Unit)? = null
    var onNext: ((View) -> Boolean)? = null

    private fun getCardNumber() = CardNumberFormatter.normalize(viewBinding?.cardNumberInput?.text)
    private fun getExpireDate() = viewBinding?.expiryDateInput?.text.orEmpty()
    private fun getCvc() = viewBinding?.cvcInput?.text.orEmpty()

    private var scannedCardCallback: ((ScannedCardResult) -> Unit)? = { it: ScannedCardResult ->
        when (it) {
            is ScannedCardResult.Success -> {
                val currentFragment = WeakReference(this)
                currentFragment.get()?.handleCardScannerSuccessResult(it)
                Unit
            }
            is ScannedCardResult.Cancel -> Unit
            is ScannedCardResult.Failure -> Unit
        }
    }
    private var clearOrScanButton: BaubleClearOrScanButton? = null
    private var cardScannerWrapper: CardScannerWrapper? = null

    override fun onAttach(context: Context) {
        super.onAttach(context)
        cardScannerWrapper = scannedCardCallback?.let { CardScannerWrapper(requireActivity(), it) }
        cardScannerWrapper?.cameraCardScannerContract = paymentOptions.features.cameraCardScannerContract
        clearOrScanButton?.update()
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        inflater.inflate(R.layout.acq_fragment_card_data_input, container, false)
        val binding = AcqFragmentCardDataInputBinding.inflate(inflater, container, false)
        this.viewBinding = binding

        binding.apply {
            val useSecureKeyboard = paymentOptions.features.useSecureKeyboard
            cardNumberInput.useSecureKeyboard = useSecureKeyboard
            expiryDateInput.useSecureKeyboard = useSecureKeyboard
            cvcInput.useSecureKeyboard = useSecureKeyboard
        }

        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        viewBinding?.apply {
            BaubleCardLogo().attach(cardNumberInput)
            val clearOrScanButton = BaubleClearOrScanButton()
            clearOrScanButton.attach(cardNumberInput, cardScannerWrapper)
            this@CardDataInputFragment.clearOrScanButton = clearOrScanButton
            val cardNumberFormatter = CardNumberFormatter()
            cardNumberInput.editText.addTextChangedListener(cardNumberFormatter)

            cardNumberInput.editText.afterTextChanged {
                cardNumberInput.editText.errorHighlighted = false

                val cardNumber = getCardNumber()
                val paymentSystem = CardPaymentSystem.resolve(cardNumber)

                if (cardNumber.startsWith("0")) {
                    cardNumberInput.editText.errorHighlighted = true
                    return@afterTextChanged
                }

                if (cardNumber.length in paymentSystem.range) {
                    if (!CardValidator.validateCardNumber(cardNumber)) {
                        cardNumberInput.editText.errorHighlighted = true
                    } else if (CardValidator.shouldAutoSwitchFromCardNumber(cardNumber)) {
                        viewBinding?.expiryDateInput?.requestViewFocus()
                    }
                }

                onDataChanged()
            }

            cardNumberInput.nextPressedListener = { onNext?.invoke(viewBinding?.cardNumberInput!!) ?: false }

            with(expiryDateInput) {
                this.useSecureKeyboard = useSecureKeyboard
                BaubleClearButton().attach(this)
                MaskFormatWatcher(createExpiryDateMask()).installOn(editText)

                editText.afterTextChanged {
                    editText.errorHighlighted = false

                    val expiryDate = getExpireDate()
                    if (expiryDate.length >= EXPIRY_DATE_MASK.length) {
                        if (CardValidator.validateExpireDate(expiryDate, false)) {
                            cvcInput.requestViewFocus()
                        } else {
                            editText.errorHighlighted = true
                        }
                    }

                    onDataChanged()
                }

                nextPressedListener = onNext?.let { { it(expiryDateInput) } }
            }

            with(cvcInput) {
                this.useSecureKeyboard = useSecureKeyboard
                editText.letterSpacing = 0.1f
                BaubleClearButton().attach(this)
                transformationMethod = PasswordTransformationMethod()
                MaskFormatWatcher(createCvcMask()).installOn(editText)

                editText.afterTextChanged {
                    editText.errorHighlighted = false
                    onDataChanged()
                }

                nextPressedListener = onNext?.let { { it(cvcInput) } }
            }

            savedInstanceState?.run {
                cardNumberInput.text = getString(SAVE_CARD_NUMBER, getCardNumber())
                expiryDateInput.text = getString(SAVE_EXPIRY_DATE, getExpireDate())
                cvcInput.text = getString(SAVE_CVC, getCvc())
            }
        }

        onDataChanged()
    }

    override fun onDestroyView() {
        super.onDestroyView()
        cardScannerWrapper?.clear()
        viewBinding = null
        clearOrScanButton = null
        cardScannerWrapper = null
        scannedCardCallback = null
    }

    override fun onSaveInstanceState(outState: Bundle) {
        with(outState) {
            putString(SAVE_CARD_NUMBER, getCardNumber())
            putString(SAVE_EXPIRY_DATE, getExpireDate())
            putString(SAVE_CVC, getCvc())
        }
    }

    fun validate(): Boolean {
        var result = true
        viewBinding?.let {
            if (CardValidator.validateCardNumber(getCardNumber())) {
                it.cardNumberInput.errorHighlighted = true
                result = false
            }
            if (CardValidator.validateExpireDate(getExpireDate(), false)) {
                it.expiryDateInput.errorHighlighted = true
                result = false
            }
            if (CardValidator.validateSecurityCode(getCvc())) {
                it.cvcInput.errorHighlighted = true
                result = false
            }
        }
        return result
    }

    fun isValid(): Boolean = CardValidator.validateCardNumber(getCardNumber()) &&
            CardValidator.validateExpireDate(getExpireDate(), false) &&
            CardValidator.validateSecurityCode(getCvc())

    private fun onDataChanged() {
        val listener = (parentFragment as? OnCardDataChanged) ?: (activity as? OnCardDataChanged)
        listener?.let {
            val data = OnCardDataChanged.Data(
                pan = getCardNumber(),
                expiryDate = getExpireDate(),
                securityCode = getCvc()
            )
            val isValid = isValid()
            it.onCardDataChanged(data, isValid)
        }
    }

    fun clearInput() {
        viewBinding?.apply {
            cardNumberInput.text = ""
            expiryDateInput.text = ""
            cvcInput.text = ""
        }
    }

    fun clearFocus() {
        viewBinding?.apply {
            cardNumberInput.clearFocus()
            expiryDateInput.clearFocus()
            cvcInput.clearFocus()
        }
    }

    private fun handleCardScannerSuccessResult(result: ScannedCardResult.Success) {
        val viewBinding = viewBinding ?: return

        val scannedCardNumber = result.data.cardNumber
        viewBinding.cardNumberInput.text = scannedCardNumber

        val scannedDate = result.data.expireDate
        viewBinding.expiryDateInput.text = scannedDate

        when {
            scannedCardNumber.isBlank() -> {
                viewBinding.cardNumberInput.requestViewFocus()
            }

            scannedDate.isBlank() -> {
                viewBinding.expiryDateInput.requestViewFocus()
            }

            else -> {
                viewBinding.cvcInput.requestViewFocus()
            }
        }
    }

    override fun onDetach() {
        super.onDetach()
        cardScannerWrapper = null
    }

    fun requestFocusOnCardNumber() {
        viewBinding?.cardNumberInput?.requestViewFocus()
    }

    fun isCvcInput(view: View): Boolean {
        return view == viewBinding?.cvcInput
    }

    fun interface OnCardDataChanged {
        fun onCardDataChanged(data: Data, isValid: Boolean)

        data class Data(
            val pan: String,
            val expiryDate: String,
            val securityCode: String
        )
    }

    companion object {

        const val EXPIRY_DATE_MASK = "__/__"
        const val CVC_MASK = "___"

        private const val SAVE_CARD_NUMBER = "extra_card_number"
        private const val SAVE_EXPIRY_DATE = "extra_expiry_date"
        private const val SAVE_CVC = "extra_save_cvc"
        private const val ARGUMENT_OPTIONS = "options"

        fun newInstance(options: BaseAcquiringOptions): CardDataInputFragment {
            return CardDataInputFragment().apply {
                arguments = bundleOf(
                    ARGUMENT_OPTIONS to options
                )
            }
        }

        fun createExpiryDateMask(): MaskImpl = MaskImpl
            .createTerminated(UnderscoreDigitSlotsParser().parseSlots(EXPIRY_DATE_MASK))

        fun createCvcMask(): MaskImpl = MaskImpl
            .createTerminated(UnderscoreDigitSlotsParser().parseSlots(CVC_MASK))
    }
}

