package ru.tinkoff.acquiring.sdk.redesign.common.cardpay

import android.view.View
import android.view.ViewGroup
import androidx.core.view.isVisible
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqCardPayComponentBinding
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.common.carddatainput.CvcComponentFocusDecorator
import ru.tinkoff.acquiring.sdk.redesign.common.emailinput.EmailInputComponent
import ru.tinkoff.acquiring.sdk.redesign.payment.model.CardChosenModel
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.ChosenCardComponent
import ru.tinkoff.acquiring.sdk.ui.component.UiComponent
import ru.tinkoff.acquiring.sdk.ui.customview.LoaderButton
import ru.tinkoff.acquiring.sdk.utils.KeyboardVisionUtils
import ru.tinkoff.acquiring.sdk.utils.setDisabledFieldAlpha
import java.lang.ref.WeakReference

/**
 * Created by i.golovachev
 */
class CardPayComponent(
    private val root: WeakReference<ViewGroup>,
    private val initingFocusAndKeyboard: Boolean = false,
    private var viewBinding: AcqCardPayComponentBinding?,
    private val email: String?,
    private val onCvcCompleted: (String) -> Unit = {},
    private val onEmailInput: (String) -> Unit = {},
    private var onEmailVisibleChange: ((Boolean) -> Unit)? = {},
    private val onChooseCardClick: () -> Unit = {},
    private val onPayClick: () -> Unit = {},
    private val onChangeCard: (CardChosenModel) -> Unit = {},
    private val onFocusCvc: View.() -> Unit = {
        requestFocus()
        isEnabled = true
    },
    useSecureKeyboard: Boolean = false,
) : UiComponent<CardChosenModel> {

    private var loaderButton: LoaderButton? = viewBinding?.loaderButton?.apply {
        setOnClickListener { onPayClick() }
    }

    private var cardCvc: CvcComponentFocusDecorator? = CvcComponentFocusDecorator(
        WeakReference(root.get()?.findViewById(R.id.cvc_container)),
        initingFocusAndKeyboard,
        onFocusCvc,
        onInputComplete = { s ->
            onCvcCompleted(s)
        },
        onDataChange = { b, s ->
            onCvcCompleted(s)
        },
        useSecureKeyboard = useSecureKeyboard
    )


    private var emailInputComponent: EmailInputComponent? = WeakReference(viewBinding?.emailInput?.root).get()?.let {
        EmailInputComponent(
            it,
            onEmailChange = { onEmailInput(it) },
            onEmailVisibleChange = { onEmailVisibleChange?.let { it1 -> it1(it) } }
        ).apply {
            render(EmailInputComponent.State(email, email != null))
        }
    }

    private var savedCardComponent: ChosenCardComponent? = viewBinding?.chosenCard?.root?.let {
        ChosenCardComponent(
            it,
            onCvcCompleted = { cvc, _ -> onCvcCompleted(cvc) },
            onChangeCard = { onChooseCardClick() },
            onFocusCvc = { viewBinding?.chosenCard?.cvcContainer },
            useSecureKeyboard = useSecureKeyboard
        )
    }

    fun render(state: CardChosenModel, email: String?, paymentOptions: PaymentOptions) {
        emailInputComponent?.render(email, email.isNullOrBlank().not())
        savedCardComponent?.render(state)
        loaderButton?.text = viewBinding?.root?.resources?.getString(
            R.string.acq_cardpay_pay,
            paymentOptions.order.amount.toHumanReadableString()
        ).orEmpty()
    }

    fun renderNewCard(state: CardChosenModel) {
        savedCardComponent?.render(state)
    }

    fun renderEnable(isEnable: Boolean) {
        loaderButton?.isEnabled = isEnable
    }

    fun renderLoader(isLoading: Boolean) {
        viewBinding?.chosenCard?.root?.setDisabledFieldAlpha(isLoading)
        loaderButton?.isLoading = isLoading
        loaderButton?.isClickable = !isLoading
    }

    fun renderInputCvc(card: CardChosenModel, paymentOptions: PaymentOptions) {
        viewBinding?.emailInput?.root?.isVisible = false
        savedCardComponent?.renderCvcOnly(card)
        savedCardComponent?.enableCvc(true)
        loaderButton?.text = viewBinding?.root?.resources?.getString(
            R.string.acq_cardpay_pay,
            paymentOptions.order.amount.toHumanReadableString()
        ).orEmpty()
    }

    fun isVisible(isVisible: Boolean) {
        viewBinding?.root?.isVisible = isVisible
    }

    fun isEnable(isEnable: Boolean) {
        savedCardComponent?.enableCvc(isEnable)
        viewBinding?.root?.isEnabled = isEnable
        emailInputComponent?.isEnable(isEnable)
    }

    fun isKeyboardVisible(isVisible: Boolean) {
        if (isVisible)
            viewBinding?.root?.let { KeyboardVisionUtils.showKeyboard(it) }
        else
            viewBinding?.root?.let { KeyboardVisionUtils.hideKeyboard(it) }
    }

    override fun render(state: CardChosenModel) {
        root.get()?.setOnClickListener { onChangeCard(state) }
    }

    fun clearCvc() {
        cardCvc?.render(null)
    }

    fun focusCvc() {
        cardCvc?.focusCvc()
    }

    fun clear() {
        viewBinding = null
        loaderButton = null
        cardCvc = null
        emailInputComponent = null
        savedCardComponent = null
        onEmailVisibleChange = null
    }

    fun clearCvcFocus() {
        cardCvc?.clearCvcFocus()
    }
}
