package ru.tinkoff.acquiring.sdk.redesign.common.emailinput

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.core.widget.doAfterTextChanged
import androidx.fragment.app.Fragment
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.smartfield.AcqTextFieldView
import ru.tinkoff.acquiring.sdk.smartfield.BaubleClearButton
import ru.tinkoff.acquiring.sdk.utils.getParent

/**
 * Created by Michael Babayan
 */
class EmailInputFragment : Fragment(), IsolatedKoinComponent {

    private var emailInput: AcqTextFieldView? = null
    private var baubleClearButton: BaubleClearButton? = null

    val viewModel by viewModel<EmailInputViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?
    ): View? = inflater.inflate(R.layout.acq_fragment_email_input, container, false)

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        emailInput = view.findViewById(R.id.email_input)

        emailInput?.let { input ->
            baubleClearButton = BaubleClearButton().apply {
                attach(input)
            }

            input.editText.doAfterTextChanged { text ->
                val cursorPos = input.editText.selectionStart
                viewModel.emailChanged(text.toString(), cursorPos)
            }

            input.onFocusChangeListener = View.OnFocusChangeListener { _, hasFocus ->
                if (hasFocus.not()) {
                    viewModel.checkValidation()
                }
            }
        }

        with(lifecycleScope) {
            launchWhenResumed {
                handleEmailState()
            }
            launchWhenResumed {
                handleEmailChangeState()
            }
            launchWhenResumed {
                handleOneTimeEvents()
            }
        }
    }

    private suspend fun handleEmailChangeState() {
        viewModel.viewState.collect { state ->
            emailInput?.let {
                with(it.editText) {
                    isEnabled = state.emailEnabled
                }
            }
        }
    }

    private suspend fun handleEmailState() {
        viewModel.combineEmailAndCursor().collect { (email, cursorPos, errorHighlighted) ->
            emailInput?.let {
                with(it.editText) {
                    setText(email)
                    setSelection(cursorPos)
                }
            }

            emailInput?.errorHighlighted = errorHighlighted
            notifyDataChanged(emailInput?.text.orEmpty(), !errorHighlighted)
        }
    }

    private suspend fun handleOneTimeEvents() {
        viewModel.oneTimeEvents.collect { state ->
            state.hideEmailInputKeyboard?.data?.takeIf { it }?.let {
                emailInput?.hideKeyboard()
            }
            state.clearEmailInputFocus?.data?.takeIf { it }?.let {
                emailInput?.clearViewFocus()
            }
            state.requestEmailInputFocus?.data?.takeIf { it }?.let {
                emailInput?.requestFocus()
            }
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        baubleClearButton?.detach()
        emailInput = null
    }

    fun withArguments(email: String?): EmailInputFragment = apply {
        arguments = bundleOf(EMAIL_ARG to email)
    }

    private fun notifyDataChanged(email: String, isValid: Boolean) {
        getParent<OnEmailDataChanged>()?.onEmailDataChanged(email, isValid)
    }

    fun interface OnEmailDataChanged {
        fun onEmailDataChanged(email: String, isValid: Boolean)
    }

    companion object {
        const val EMAIL_ARG = "EMAIL_ARG"
        fun getInstance(email: String?) = EmailInputFragment().withArguments(email)
    }
}