package ru.tinkoff.acquiring.sdk.redesign.main

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.os.Parcelable
import android.view.View
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.core.graphics.Insets
import androidx.core.view.ViewCompat
import androidx.core.view.WindowCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.updatePadding
import androidx.fragment.app.Fragment
import androidx.lifecycle.createSavedStateHandle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.viewmodel.initializer
import androidx.lifecycle.viewmodel.viewModelFactory
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.android.ext.android.inject
import org.koin.android.scope.AndroidScopeComponent
import org.koin.core.qualifier.named
import org.koin.core.scope.Scope
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.RunMode
import ru.tinkoff.acquiring.sdk.databinding.AcqSingleActivityBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.di.Scopes
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringExceptionWithErrorCode
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringExceptionWithPaymentId
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.AttachCardOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.BaseAcquiringOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.vm.MainPaymentFormFlowViewModel
import ru.tinkoff.acquiring.sdk.redesign.mainform.ui.MainPaymentFormFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.mirpay.ui.MirPayFragment
import ru.tinkoff.acquiring.sdk.redesign.mirpay.ui.MirPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.PaymentByCardFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.recurrent.ui.RecurrentPaymentFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.recurrent.ui.RecurrentPaymentFlowViewModel
import ru.tinkoff.acquiring.sdk.redesign.sbp.ui.SbpPaymentFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.tpay.ui.TPayResult
import ru.tinkoff.acquiring.sdk.redesign.tpay.ui.TpayFragment
import ru.tinkoff.acquiring.sdk.toggles.DeviceIdProviderImpl
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.toggles.toggles.PaymentByCardV2FeatureToggle
import ru.tinkoff.acquiring.sdk.ui.fragments.AttachCardFlowFragment


/**
 * @author s.y.biryukov
 */
class SingleActivity : AppCompatActivity(), AndroidScopeComponent, IsolatedKoinComponent {
    private lateinit var viewBinding: AcqSingleActivityBinding

    override val scope: Scope = getKoin().getOrCreateScope(
        Scopes.SingleActivityScope.name, named(Scopes.SingleActivityScope.name))

    private val viewModel: SingleActivityViewModel by viewModels {
        viewModelFactory {
            initializer {
                SingleActivityViewModel(
                    savedStateHandle = createSavedStateHandle(),
                    featureToggleManager = FeatureToggleManager.getInstance(this@SingleActivity),
                    deviceIdProvider = DeviceIdProviderImpl(this@SingleActivity)
                )
            }
        }
    }

    private val featureToggleManager: FeatureToggleManager by inject()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        this.viewBinding = AcqSingleActivityBinding.inflate(layoutInflater)
        setContentView(viewBinding.root)

        WindowCompat.setDecorFitsSystemWindows(window, false)
        ViewCompat.setOnApplyWindowInsetsListener(viewBinding.root) { v: View, windowInsets: WindowInsetsCompat ->
            val insets: Insets = windowInsets.getInsets(WindowInsetsCompat.Type.systemBars())
            v.updatePadding(
                top = insets.top
            )
            windowInsets
        }

        registerTpayPaymentResultListener()
        resisterSbpPaymentResultListener()
        registerMirPaymentResultListener()
        registerMainPaymentResultListener()
        registerCardPaymentResultListener()
        registerRecurrentPaymentResultListener()
        registerAttachCardListener()


        lifecycleScope.launch {
            viewModel.commandFlow.collect {
                when (it) {
                    is SingleActivityViewModel.Command.StartFlow -> startFlow(it)
                }
            }
        }
    }

    override fun onStart() {
        super.onStart()
        overridePendingTransition(0, 0)
    }

    override fun onResume() {
        super.onResume()
        viewModel.handleEvent(SingleActivityViewModel.Event.Resume)
    }

    override fun onPause() {
        super.onPause()
        overridePendingTransition(0 ,0)
    }

    override fun onDestroy() {
        super.onDestroy()
        scope.close()
        setSupportActionBar(null)
    }

    private fun startFlow(it: SingleActivityViewModel.Command.StartFlow) {
        when (val arguments = it.arguments) {
            is Arguments.TpayPaymentArguments -> startTpayPayment(arguments)
            is Arguments.SbpPaymentArguments -> startSbpPayment(arguments)
            is Arguments.MirPayPaymentArguments -> startMirPayPayment(arguments)
            is Arguments.MainPaymentArguments -> startMainPayment(arguments)
            is Arguments.PaymentByCardArguments -> startCardPayment(arguments)
            is Arguments.RecurrentPaymentArguments -> startRecurrentPayment(arguments)
            is Arguments.AttachCardArguments -> startAttachCard(arguments)
        }
    }

    private fun startRecurrentPayment(arguments: Arguments.RecurrentPaymentArguments) {
        val fragment = RecurrentPaymentFlowFragment.newInstance(
            paymentOptions = arguments.options,
            card = arguments.card
        )

        showFragment(fragment)
    }

    private fun startAttachCard(arguments: Arguments.AttachCardArguments) {
        val fragment = AttachCardFlowFragment.newInstance(
            options = arguments.options,
            showError = arguments.showError,
            isNewCardViewToggleEnabled = false,
        )

        showFragment(fragment)
    }

    private fun registerRecurrentPaymentResultListener() {
        RecurrentPaymentFlowFragment.registerResultListener(supportFragmentManager, this) {
            when (it) {
                RecurrentPaymentFlowViewModel.Result.Cancelled -> {
                    returnResult(Result.RecurrentPaymentResult.Cancel)
                }

                is RecurrentPaymentFlowViewModel.Result.FinishError -> {
                    returnResult(
                        Result.RecurrentPaymentResult.Error(
                            error = it.error,
                            paymentId = it.paymentId,
                        )
                    )
                }

                is RecurrentPaymentFlowViewModel.Result.Success -> {
                    returnResult(
                        Result.RecurrentPaymentResult.Success(
                            paymentId = it.paymentId,
                            rebillId = it.rebillId
                        )
                    )
                }
            }
        }
    }

    private fun registerAttachCardListener() {
        AttachCardFlowFragment.registerResultListener(supportFragmentManager, this) {
            when (it) {
                is AttachCardFlowFragment.Result.Success -> returnResult(Result.AttachCardResult.Success(
                    panSuffix = it.panSuffix,
                    cardId = it.cardId
                ))
                is AttachCardFlowFragment.Result.Failed -> returnResult(Result.AttachCardResult.Error(it.error))
                AttachCardFlowFragment.Result.Cancel -> returnResult(Result.AttachCardResult.Cancel)
            }
        }
    }

    private fun startCardPayment(arguments: Arguments.PaymentByCardArguments) {
        lifecycleScope.launch {
            val fragment = PaymentByCardFlowFragment.newInstance(
                paymentOptions = arguments.options,
                card = arguments.card,
                showArrow = arguments.showArrow,
                isNewCardViewToggleEnabled = featureToggleManager.isEnabled(PaymentByCardV2FeatureToggle)
            )

            showFragment(fragment)
        }
    }

    private fun registerCardPaymentResultListener() {
        PaymentByCardFlowFragment.registerResultListener(supportFragmentManager, this) {
            when (it) {
                is PaymentByCardFlowFragment.Result.PaymentSuccess -> {
                    returnResult(
                        Result.PaymentByCardResult.Success(
                            paymentId = it.paymentId,
                            cardId = it.cardId,
                            rebillId = it.rebillId,
                        )
                    )
                }

                is PaymentByCardFlowFragment.Result.PaymentError -> {
                    returnResult(
                        Result.PaymentByCardResult.Error(
                            error = it.error,
                            paymentId = it.paymentId
                        )
                    )
                }

                PaymentByCardFlowFragment.Result.PaymentCancel -> {
                    returnResult(
                        Result.PaymentByCardResult.Cancel
                    )
                }
            }
        }
    }

    private fun startMainPayment(arguments: Arguments.MainPaymentArguments) {
        val fragment = MainPaymentFormFlowFragment.newInstance(
            paymentOptions = arguments.options.apply {
                sdkContext.runMode = RunMode.FORM
            }
        )

        showFragment(fragment)
    }

    private fun registerMainPaymentResultListener() {
        MainPaymentFormFlowFragment.registerResultListener(supportFragmentManager, this) {
            when (it) {
                MainPaymentFormFlowViewModel.Result.FinishCancelled -> {
                    returnResult(
                        Result.MainPaymentResult.Cancel
                    )
                }

                is MainPaymentFormFlowViewModel.Result.FinishError -> {
                    val error = it.error
                    returnResult(
                        Result.MainPaymentResult.Error(
                            error = error,
                            errorCode = (error as? AcquiringExceptionWithErrorCode)?.errorCode?.toIntOrNull(),
                            paymentId = (error as? AcquiringExceptionWithPaymentId)?.paymentId,
                        )
                    )
                }

                is MainPaymentFormFlowViewModel.Result.FinishSuccess -> {
                    returnResult(
                        Result.MainPaymentResult.Success(
                            paymentId = it.paymentId,
                            cardId = it.cardId,
                            rebillId = it.rebillId
                        )
                    )
                }
            }
        }
    }

    private fun startMirPayPayment(arguments: Arguments.MirPayPaymentArguments) {
        val fragment = MirPayFragment.newInstance(
            paymentOptions = arguments.options
        )

        showFragment(fragment)
    }

    private fun registerMirPaymentResultListener() {
        MirPayFragment.registerFragmentResultListener(supportFragmentManager, this) { result ->
            when (result) {
                is MirPaymentResult.Success -> {
                    returnResult(
                        Result.MirPayPaymentResult.Success(
                            paymentId = result.paymentId
                        )
                    )
                }

                is MirPaymentResult.Error -> {
                    returnResult(
                        Result.MirPayPaymentResult.Error(
                            error = result.error,
                            errorCode = result.errorCode?.toIntOrNull(),
                            paymentId = result.paymentId,
                        )
                    )
                }

                is MirPaymentResult.Canceled -> {
                    returnResult(Result.MirPayPaymentResult.Cancel)
                }
            }
        }
    }

    private fun startSbpPayment(arguments: Arguments.SbpPaymentArguments) {
        val flowFragment = SbpPaymentFlowFragment.newInstance(
            paymentOptions = arguments.options
        )
        showFragment(flowFragment)
    }

    private fun resisterSbpPaymentResultListener() {
        SbpPaymentFlowFragment.registerResultListener(supportFragmentManager, this) {
            when (it) {
                SbpPaymentFlowFragment.Result.Cancel -> returnResult(Result.SbpPaymentResult.Cancel)
                is SbpPaymentFlowFragment.Result.Error -> returnResult(
                    Result.SbpPaymentResult.Error(
                        errorCode = it.errorCode,
                        error = it.error,
                    )
                )

                is SbpPaymentFlowFragment.Result.Success -> returnResult(
                    Result.SbpPaymentResult.Success(
                        paymentId = it.paymentId
                    )
                )
            }
        }
    }

    private fun startTpayPayment(arguments: Arguments.TpayPaymentArguments) {
        val tpayFlowFragment = TpayFragment.newInstance(
            paymentOptions = arguments.options,
            version = arguments.version
        )
        showFragment(tpayFlowFragment)
    }

    private fun registerTpayPaymentResultListener() {
        TpayFragment.setFragmentResultListener(supportFragmentManager, this) {
            when (it) {
                TPayResult.Canceled -> returnResult(Result.TpayPaymentResult.Canceled)
                is TPayResult.Error -> returnResult(
                    Result.TpayPaymentResult.Error(
                        error = it.error,
                        paymentId = it.paymentId,
                        errorCode = it.errorCode
                    )
                )

                is TPayResult.Success -> returnResult(
                    Result.TpayPaymentResult.Success(
                        paymentId = it.paymentId
                    )
                )
            }
        }
    }

    private fun showFragment(fragment: Fragment) {
        supportFragmentManager.beginTransaction()
            .replace(R.id.acq_tpay_root, fragment)
            .commit()
    }

    private fun returnResult(result: Result) {
        val resultCode = if (result is ResultType.Success) {
            Activity.RESULT_OK
        } else {
            Activity.RESULT_CANCELED
        }
        setResult(resultCode, Intent().putExtra(RESULT_KEY, result))
        finish()
    }

    companion object {
        const val EXTRA_KEY = "SingleActivity.EXTRA_KEY"
        private const val RESULT_KEY = "SingleActivity.RESULT_KEY"
        fun createIntent(context: Context, arguments: Arguments): Intent {
            return Intent(context, SingleActivity::class.java)
                .putExtra(EXTRA_KEY, arguments)
        }

        fun <T: Result> getResult(intent: Intent?): T? {
            return intent?.getParcelableExtra(RESULT_KEY) as? T
        }
    }

    sealed interface Arguments : Parcelable {
        val options: BaseAcquiringOptions

        @Parcelize
        data class RecurrentPaymentArguments(
            override val options: PaymentOptions,
            val card: Card,
        ) : Arguments

        @Parcelize
        data class TpayPaymentArguments(
            override val options: PaymentOptions,
            val version: String,
        ) : Arguments

        @Parcelize
        data class SbpPaymentArguments(
            override val options: PaymentOptions
        ) : Arguments

        @Parcelize
        data class MirPayPaymentArguments(
            override val options: PaymentOptions
        ) : Arguments

        @Parcelize
        data class MainPaymentArguments(
            override val options: PaymentOptions
        ) : Arguments

        @Parcelize
        class PaymentByCardArguments(
            override val options: PaymentOptions,
            val card: Card? = null,
            val showArrow: Boolean = false
        ) : Arguments

        @Parcelize
        class AttachCardArguments(
            override val options: AttachCardOptions,
            val showError: Boolean,
        ) : Arguments
    }

    sealed interface ResultType {
        interface Success : ResultType
        interface Cancelled : ResultType
        interface Error : ResultType
    }

    sealed interface Result : Parcelable {
        interface TpayPaymentResult : Result {
            @Parcelize
            class Success(
                val paymentId: Long,
                val cardId: String? = null,
                val rebillId: String? = null
            ) : TpayPaymentResult, ResultType.Success

            @Parcelize
            object Canceled : TpayPaymentResult, ResultType.Cancelled

            @Parcelize
            class Error(
                val error: Throwable,
                val errorCode: Int?,
                val paymentId: Long?
            ) : TpayPaymentResult, ResultType.Error
        }

        interface SbpPaymentResult : Result {
            @Parcelize
            data object Cancel : SbpPaymentResult, ResultType.Cancelled

            @Parcelize
            data class Success(
                val paymentId: Long,
            ) : SbpPaymentResult, ResultType.Success

            @Parcelize
            data class Error(
                val error: Throwable,
                val errorCode: Int? = null,
            ) : SbpPaymentResult, ResultType.Error
        }

        interface MirPayPaymentResult : Result {
            @Parcelize
            data object Cancel : MirPayPaymentResult, ResultType.Cancelled

            @Parcelize
            data class Success(
                val paymentId: Long,
            ) : MirPayPaymentResult, ResultType.Success

            @Parcelize
            data class Error(
                val error: Throwable,
                val errorCode: Int? = null,
                val paymentId: Long? = null,
            ) : MirPayPaymentResult, ResultType.Error
        }

        interface MainPaymentResult : Result {
            @Parcelize
            data object Cancel : MainPaymentResult, ResultType.Cancelled

            @Parcelize
            data class Success(
                val paymentId: Long,
                val cardId: String? = null,
                val rebillId: String? = null
            ) : MainPaymentResult, ResultType.Success

            @Parcelize
            data class Error(
                val error: Throwable,
                val errorCode: Int? = null,
                val paymentId: Long? = null,
            ) : MainPaymentResult, ResultType.Error
        }

        interface PaymentByCardResult : Result {
            @Parcelize
            data object Cancel : PaymentByCardResult, ResultType.Cancelled

            @Parcelize
            data class Success(
                val paymentId: Long,
                val cardId: String? = null,
                val rebillId: String? = null
            ) : PaymentByCardResult, ResultType.Success

            @Parcelize
            data class Error(
                val error: Throwable,
                val errorCode: Int? = null,
                val paymentId: Long? = null,
            ) : PaymentByCardResult, ResultType.Error
        }

        interface RecurrentPaymentResult : Result {
            @Parcelize
            data object Cancel : RecurrentPaymentResult, ResultType.Cancelled

            @Parcelize
            data class Success(
                val paymentId: Long,
                val rebillId: String? = null
            ) : RecurrentPaymentResult, ResultType.Success

            @Parcelize
            data class Error(
                val error: Throwable,
                val errorCode: Int? = null,
                val paymentId: Long? = null,
            ) : RecurrentPaymentResult, ResultType.Error
        }

        interface AttachCardResult : Result {
            @Parcelize
            data object Cancel : AttachCardResult, ResultType.Cancelled

            @Parcelize
            data class Success(
                val cardId: String,
                val panSuffix: String
            ) : AttachCardResult, ResultType.Success

            @Parcelize
            data class Error(
                val error: Throwable,
                val errorCode: Int? = null,
                val paymentId: Long? = null,
            ) : AttachCardResult, ResultType.Error
        }
    }
}
