package ru.tinkoff.acquiring.sdk.redesign.main

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.toggles.DeviceIdProvider
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager

/**
 * @author s.y.biryukov
 */
class SingleActivityViewModel(
    private val savedStateHandle: SavedStateHandle,
    private val featureToggleManager: FeatureToggleManager,
    private var deviceIdProvider: DeviceIdProvider
): ViewModel() {
    private var isResumeFirstTime: Boolean = true

    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow: SharedFlow<Command> = _commandFlow.asSharedFlow()

    fun handleEvent(event: Event) {
        when(event) {
            Event.Resume -> onResume()
        }
    }

    private fun onResume() {
        if (isResumeFirstTime) {
            val arguments = savedStateHandle.get<SingleActivity.Arguments>(SingleActivity.EXTRA_KEY) ?: return
            val terminalKey = requireNotNull(arguments.options.terminalKey) { "terminalKey is null" }
            updateFeatureToggleContext(terminalKey)
            isResumeFirstTime = false

            viewModelScope.launch {
                _commandFlow.emit(Command.StartFlow(arguments))
            }
        }
    }

    private fun updateFeatureToggleContext(terminalKey: String) {
        viewModelScope.launch {
            featureToggleManager.updateFeatureToggleContext {
                it.copy(
                    terminalKey = terminalKey,
                    deviceId = deviceIdProvider.getDeviceId(),
                    appVersion = AcquiringSdk.appVersion,
                )
            }
        }
    }

    sealed interface Event{
        data object Resume : Event
    }

    sealed interface Command{
        class StartFlow(
            val arguments: SingleActivity.Arguments
        ) : Command
    }
}
