package ru.tinkoff.acquiring.sdk.redesign.mainform

import android.content.Context
import android.content.Intent
import android.os.Parcelable
import androidx.activity.result.contract.ActivityResultContract
import kotlinx.parcelize.Parcelize
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringApiException
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.common.result.AcqPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.main.SingleActivity
import ru.tinkoff.acquiring.sdk.redesign.main.SingleActivity.Result.MainPaymentResult

object MainFormLauncher {
    sealed class Result
    class Success(
        override val paymentId: Long,
        override val cardId: String? = null,
        override val rebillId: String? = null
    ) : Result(), AcqPaymentResult.Success

    object Canceled : Result(), AcqPaymentResult.Canceled
    class Error(
        override val error: Throwable,
        override val errorCode: Int?,
        override val paymentId: Long?
    ) : Result(), AcqPaymentResult.Error {

        constructor(error: AcquiringApiException) : this(
            error,
            error.response?.errorCode?.toInt(),
            null
        )
    }

    @Parcelize
    class StartData(val paymentOptions: PaymentOptions) : Parcelable

    object Contract : ActivityResultContract<StartData, Result>() {
        override fun createIntent(context: Context, input: StartData): Intent {
            return SingleActivity.createIntent(
                context,
                SingleActivity.Arguments.MainPaymentArguments(input.paymentOptions)
            )
        }

        override fun parseResult(resultCode: Int, intent: Intent?): Result {
            return when (val result: MainPaymentResult? = SingleActivity.getResult(intent)) {
                is MainPaymentResult.Error -> {
                    Error(
                        error = result.error,
                        errorCode = result.errorCode,
                        paymentId = result.paymentId,
                    )
                }

                is MainPaymentResult.Success -> {
                    Success(
                        paymentId = result.paymentId,
                        cardId = result.cardId,
                        rebillId = result.rebillId
                    )
                }

                else -> Canceled
            }
        }
    }
}
