package ru.tinkoff.acquiring.sdk.redesign.mainform.navigation

import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.asSharedFlow
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.ThreeDsState
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ChooseCardLauncher
import ru.tinkoff.acquiring.sdk.redesign.common.result.AcqPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.mirpay.MirPayLauncher
import ru.tinkoff.acquiring.sdk.redesign.payment.PaymentByCardLauncher
import ru.tinkoff.acquiring.sdk.redesign.sbp.SbpPayLauncher
import ru.tinkoff.acquiring.sdk.redesign.tpay.TpayLauncher
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBasedTransaction

/**
 * Created by i.golovachev
 */
internal class MainFormNavController {

    private val channelNav = MutableSharedFlow<Navigation>()
    val navFlow = channelNav.asSharedFlow()

    suspend fun toSbp(paymentOptions: PaymentOptions) = channelNav.emit(
        Navigation.ToSbp(
            SbpPayLauncher.StartData(
                paymentOptions
            )
        )
    )

    suspend fun toPayCard(paymentOptions: PaymentOptions, cards: List<Card>, chosenCard: Card?) {
        channelNav.emit(
            Navigation.ToPayByCard(
                PaymentByCardLauncher.StartData(
                    paymentOptions.apply {
                        features.selectedCardId = chosenCard?.cardId
                    },
                    ArrayList(cards),
                    withArrowBack = true
                )
            )
        )
    }

    suspend fun toChooseCard(paymentOptions: PaymentOptions, card: Card? = null) {
        val savedCardsOptions: SavedCardsOptions = SavedCardsOptions().apply {
            setTerminalParams(
                paymentOptions.terminalKey,
                paymentOptions.publicKey
            )
            customer = paymentOptions.customer
            features = paymentOptions.features
            features.selectedCardId = card?.cardId
            withArrowBack = true
            mode = SavedCardsOptions.Mode.PAYMENT
        }
        channelNav.emit(
            Navigation.ToChooseCard(
                ChooseCardLauncher.StartData(
                    savedCardsOptions = savedCardsOptions,
                    paymentOptions = paymentOptions,
                )
            )
        )
    }

    suspend fun toTpay(
        paymentOptions: PaymentOptions,
        version: String
    ) {
        channelNav.emit(Navigation.ToTpay(TpayLauncher.StartData(paymentOptions, version)))
    }

    suspend fun toTpayWebView() {
        channelNav.emit(Navigation.ToWebView(TPAY_URL))
    }

    suspend fun toMirPay(paymentOptions: PaymentOptions) {
        channelNav.emit(Navigation.ToMirPay(MirPayLauncher.StartData(paymentOptions)))
    }

    suspend fun to3ds(
        paymentOptions: PaymentOptions,
        threeDsState: ThreeDsState,
        paymentSource: PaymentSource?
    ) =
        channelNav.emit(Navigation.To3ds(paymentOptions, threeDsState, paymentSource = paymentSource))


    suspend fun close(acqPaymentResult: AcqPaymentResult) =
        channelNav.emit(Navigation.Return(acqPaymentResult))

    suspend fun toAppBaseChallenge(transaction: ThreeDsAppBasedTransaction, threedsData: ThreeDsData) =
        channelNav.emit(Navigation.ToAppBaseChallenge(transaction,  threedsData = threedsData))


    sealed interface Navigation {
        class ToSbp(val startData: SbpPayLauncher.StartData) : Navigation

        class ToPayByCard(val startData: PaymentByCardLauncher.StartData) : Navigation

        class ToChooseCard(val startData: ChooseCardLauncher.StartData) : Navigation

        class ToTpay(val startData: TpayLauncher.StartData) : Navigation

        class ToMirPay(val startData: MirPayLauncher.StartData) : Navigation

        class ToWebView(val url: String) : Navigation

        class To3ds(
            val paymentOptions: PaymentOptions,
            val threeDsState: ThreeDsState,
            val paymentSource: PaymentSource?
        ) : Navigation

        class ToAppBaseChallenge(
            val transaction: ThreeDsAppBasedTransaction,
            val threedsData: ThreeDsData
        ) : Navigation

        class Return(val result: AcqPaymentResult) : Navigation
    }

    companion object {
        private const val TPAY_URL = "https://www.tinkoff.ru/cards/debit-cards/tinkoff-pay/form/"
    }
}

