package ru.tinkoff.acquiring.sdk.redesign.mainform.presentation

import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.enums.CardStatus
import ru.tinkoff.acquiring.sdk.redesign.common.savedcard.SavedCardsRepository
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.primary.PrimaryButtonConfigurator
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.secondary.SecondButtonConfigurator
import ru.tinkoff.acquiring.sdk.responses.Paymethod
import ru.tinkoff.acquiring.sdk.responses.TerminalInfo
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.toggles.toggles.whitelabel.LogoVisibilityFeatureToggle
import ru.tinkoff.acquiring.sdk.toggles.toggles.whitelabel.MirPayVisibilityFeatureToggle
import ru.tinkoff.acquiring.sdk.toggles.toggles.whitelabel.SBPVisibilityFeatureToggle
import ru.tinkoff.acquiring.sdk.toggles.toggles.whitelabel.TPayVisibilityFeatureToggle
import ru.tinkoff.acquiring.sdk.utils.ConnectionChecker
import ru.tinkoff.acquiring.sdk.utils.checkNotNull

/**
 * Created by i.golovachev
 */
internal class MainPaymentFormFactory(
    private val sdk: AcquiringSdk,
    private val savedCardsRepository: SavedCardsRepository,
    private val primaryButtonConfigurator: PrimaryButtonConfigurator,
    private val secondButtonConfigurator: SecondButtonConfigurator,
    private val mergeMethodsStrategy: MergeMethodsStrategy,
    private val connectionChecker: ConnectionChecker,
    private val customerKey: String? = null,
    private val card: Card? = null,
    private val featureToggleManager: FeatureToggleManager
) {

    suspend fun getState(): MainPaymentForm.State {
        val terminalInfo = getTerminalInfo()
        val cards = getSavedCards(customerKey)
        return getUi(terminalInfo, cards, card)
    }

    private suspend fun getUi(t: TerminalInfo?, cards: List<Card>, selectedCard: Card?): MainPaymentForm.State {
        val targetCard = selectedCard ?: cards.firstOrNull()
        val terminalInfo = filterPaymentMethods(t)
        val primary = primaryButtonConfigurator.get(terminalInfo, targetCard)
        val secondaries = secondButtonConfigurator.get(terminalInfo)

        val ui = mergeMethodsStrategy.merge(primary, secondaries).copy(
            isLogoVisible = featureToggleManager.isEnabled(LogoVisibilityFeatureToggle)
        )

        return MainPaymentForm.State(
            ui = ui,
            data = MainPaymentForm.Data(
                terminalInfo,
                cards,
                targetCard
            ),
            noInternet = connectionChecker.isOnline().not()
        )
    }

    /**
     * Фильтруем доступные способы оплаты в соответствии с тоглами
     */
    private suspend fun filterPaymentMethods(t: TerminalInfo?): TerminalInfo? {
        if (t == null) return null
        return TerminalInfo(
            addCardScheme = t.addCardScheme,
            tokenRequired = t.tokenRequired,
            initTokenRequired = t.initTokenRequired,
            paymethods = t.paymethods.filter {
                when (it.paymethod) {
                    Paymethod.MirPay -> featureToggleManager.isEnabled(MirPayVisibilityFeatureToggle)
                    Paymethod.TinkoffPay -> featureToggleManager.isEnabled(TPayVisibilityFeatureToggle)
                    Paymethod.SBP -> featureToggleManager.isEnabled(SBPVisibilityFeatureToggle)
                    else -> true
                }
            }
        )
    }

    private suspend fun getSavedCards(customerKey: String?): List<Card> {
        if (customerKey == null) return emptyList()

        return try {
            savedCardsRepository.getCards(customerKey, true)
                .filter { it.status == CardStatus.ACTIVE }
        } catch (e: Throwable) {
            AcquiringSdk.log(e)
            emptyList()
        }
    }

    private suspend fun getTerminalInfo() : TerminalInfo {
        return sdk.getTerminalPayMethods().execute()
            .terminalInfo
            .checkNotNull { "Не удалось получить информацию о доступных способах оплаты" }
    }
}
