package ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.vm

import androidx.annotation.VisibleForTesting
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.map
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.analytics.ChosenMethod
import ru.tinkoff.acquiring.sdk.models.paysources.AttachedCard
import ru.tinkoff.acquiring.sdk.models.result.PaymentResult
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardProcess
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardState
import ru.tinkoff.acquiring.sdk.redesign.common.emailinput.models.EmailValidator
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.MainPaymentFromUtils.CHOSEN_CARD
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.MainPaymentFromUtils.CVC_KEY
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.MainPaymentFromUtils.EMAIL_KEY
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.MainPaymentFromUtils.NEED_EMAIL_KEY
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.analytics.MainFormAnalyticsDelegate
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.process.MainFormPaymentProcessMapper
import ru.tinkoff.acquiring.sdk.redesign.payment.model.CardChosenModel
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsLauncher
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.validators.CardValidator
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeResult
import ru.tinkoff.acquiring.sdk.utils.BankCaptionProvider
import ru.tinkoff.acquiring.sdk.utils.CoroutineManager

/**
 * Created by i.golovachev
 */
internal class MainFormInputCardViewModel(
    private val savedStateHandle: SavedStateHandle,
    private val byCardProcess: PaymentByCardProcess,
    private val mapper: MainFormPaymentProcessMapper,
    private val bankCaptionProvider: BankCaptionProvider,
    private val mainFormAnalyticsDelegate: MainFormAnalyticsDelegate,
    private val paymentOptions: PaymentOptions,
    private val coroutineManager: CoroutineManager,
) : ViewModel() {

    val savedCardFlow =
        savedStateHandle.getStateFlow<CardChosenModel?>(CHOSEN_CARD, null).filterNotNull()
    val cvcFlow = savedStateHandle.getStateFlow(CVC_KEY, "")
    val emailFlow = savedStateHandle.getStateFlow(EMAIL_KEY, paymentOptions.customer.email)
    val needEmail = savedStateHandle.getStateFlow(
        NEED_EMAIL_KEY,
        paymentOptions.customer.email.isNullOrBlank().not()
    )
    val payEnable = combine(cvcFlow, needEmail, emailFlow) { cvc, needEmailValidate, email ->
        validateState(cvc, needEmailValidate, email)
    }
    val isLoading =
        byCardProcess.state.map { it is PaymentByCardState.Started || it is PaymentByCardState.ThreeDsAppBase }
    val paymentStatus = byCardProcess.state.map { mapper(it) }

    private val _loadingState = MutableStateFlow<Notification?>(null)
    val loadingState = _loadingState.asStateFlow()

    fun choseCard(cardChosenModel: CardChosenModel) {
        savedStateHandle[CHOSEN_CARD] = cardChosenModel
    }

    fun choseCard(card: Card) {
        savedStateHandle[CHOSEN_CARD] = CardChosenModel(card, bankCaptionProvider(card.pan!!))
    }

    fun setCvc(cvc: String) {
        savedStateHandle[CVC_KEY] = cvc
    }

    fun needEmail(isNeed: Boolean) {
        savedStateHandle[NEED_EMAIL_KEY] = isNeed
    }

    fun email(email: String) {
        savedStateHandle[EMAIL_KEY] = email
    }

    fun set3dsResult(error: ThreeDsLauncher.Result.Error) {
        byCardProcess.set3dsResult(error)
    }

    fun set3dsResult(error: Throwable, paymentId: Long) {
        byCardProcess.set3dsResult(error, paymentId)
    }

    fun set3dsResult(paymentResult: PaymentResult) {
        byCardProcess.set3dsResult(paymentResult, paymentOptions)
    }

    fun pay() = coroutineManager.launchOnBackground {
        val card = savedStateHandle.get<CardChosenModel>(CHOSEN_CARD)!!
        val cvc = savedStateHandle.get<String>(CVC_KEY)
        val email = savedStateHandle.get<String>(EMAIL_KEY)
        val isNeedEmail = savedStateHandle.get<Boolean>(NEED_EMAIL_KEY) ?: false
        byCardProcess.start(
            paymentOptions = mainFormAnalyticsDelegate.prepareOptions(
                paymentOptions,
                ChosenMethod.Card
            ),
            cardData = AttachedCard(cardId = card.id, cvc = cvc),
            email = email.takeIf { isNeedEmail }
        )
    }

    @VisibleForTesting
    fun validateState(
        cvc: String,
        needEmailValidate: Boolean,
        email: String?
    ): Boolean {
        val validateEmailIfNeed =
            if (needEmailValidate) EmailValidator.validate(email) else true
        return CardValidator.validateSecurityCode(cvc) && validateEmailIfNeed
    }

    fun processChallengeResult(result: AppBaseChallengeResult) {
        when (result) {
            AppBaseChallengeResult.Cancelled -> {
                byCardProcess.recreate()
            }
            AppBaseChallengeResult.Loading -> {
                _loadingState.value = Notification(
                    type = StatusViewData.Type.PROGRESS,
                    title = R.string.acq_commonsheet_processing_title,
                    description = R.string.acq_commonsheet_processing_description,
                )
            }
            is AppBaseChallengeResult.ProtocolError -> {
                set3dsResult(result.error, result.paymentId)
            }

            is AppBaseChallengeResult.RuntimeError -> {
                set3dsResult(result.error, result.paymentId)
            }

            is AppBaseChallengeResult.Success -> {
                set3dsResult(result.paymentResult)
            }

            is AppBaseChallengeResult.TimeOut -> {
                set3dsResult(result.error, result.paymentId)
            }

            is AppBaseChallengeResult.Error -> {
                set3dsResult(result.error, result.paymentId)
            }
        }
    }
}
