package ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.vm

import android.os.Parcelable
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.models.result.PaymentResult
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ChooseCardLauncher
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.CardListFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.mainform.ui.MainPaymentFormResult
import ru.tinkoff.acquiring.sdk.redesign.mirpay.ui.MirPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.PaymentByCardFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.sbp.ui.SbpPaymentFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.tpay.ui.TPayResult
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsResult
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory

/**
 * @author s.y.biryukov
 */
class MainPaymentFormFlowViewModel(
    savedStateHandle: SavedStateHandle
) : ViewModel() {
    private var isResumeFirstTime: Boolean = true

    private val _stateFlow = MutableStateFlow(State())
    val stateFlow = _stateFlow.asStateFlow()

    private val _commandFlow: MutableSharedFlow<Command> = MutableSharedFlow()
    val commandFlow = _commandFlow.asSharedFlow()

    private val paymentOptions: PaymentOptions = savedStateHandle.get("paymentOptions")!!

    fun handleEvent(event: Event) {
        when (event) {
            Event.Resume -> onResume()
            is Event.MainPaymentFormResult -> handleMainPaymentFormResult(event.result)
            is Event.PaymentByCardFlowFragmentResult -> handlePayByCardResult(event.result)
            is Event.SbpPaymentFragmentResult -> handleSbpResult(event.result)
            is Event.TpayPaymentFragmentResult -> handleTpayResult(event.result)
            is Event.MirPayPaymentFragmentResult -> handleMirPayResult(event.result)
            is Event.CardListFlowFragmentResult -> handleCardListFlowResult(event.result)
            is Event.ThreeDsFlowResult -> handleThreeDsResult(event.result)
        }
    }

    private fun handleThreeDsResult(result: ThreeDsResult) {
        showEmptyScreen()
        when (result) {
            ThreeDsResult.Cancel -> finishWithCancel()
            is ThreeDsResult.Error -> onThreeDsError(result.error)
            is ThreeDsResult.Success -> onThreeDsSuccess(result)
        }
    }

    private fun showEmptyScreen() {
        viewModelScope.launch {
            _commandFlow.emit(Command.OpenEmptyScreen)
        }
    }

    private fun onThreeDsSuccess(result: ThreeDsResult.Success) {
        val onClick: () -> Unit = {
            val paymentResult = result.result as PaymentResult
            finishWithSuccess(
                paymentId = paymentResult.paymentId!!,
                cardId = paymentResult.cardId,
                rebillId = paymentResult.rebillId,
            )
        }

        if (paymentOptions.features.showPaymentNotifications) {
            _stateFlow.update {
                it.copy(
                    notification = NotificationFactory.getPaymentSuccessCommonNotification(
                        onClick = onClick,
                        amount = paymentOptions.order.amount
                    )
                )
            }
        } else {
            onClick()
        }
    }

    private fun onThreeDsError(error: Throwable) {
        val onClick: () -> Unit = {
            _stateFlow.update {
                it.copy(notification = null)
            }
            returnToMainFormOrFinish(error)
        }

        if (paymentOptions.features.showPaymentNotifications) {
            _stateFlow.update {
                it.copy(
                    notification = NotificationFactory.getPaymentErrorNotification(
                        exception = error,
                        onClickFactory = { onClick },
                        options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
                    )
                )
            }
        } else {
            onClick()
        }
    }

    private fun handleCardListFlowResult(result: CardListFlowFragment.Result) {
        when (result) {
            CardListFlowFragment.Result.Cancel -> finishWithCancel()
            is CardListFlowFragment.Result.CardChosen -> onCardChanged(result)
            is CardListFlowFragment.Result.Failed -> returnToMainFormOrFinish(result.error)
            is CardListFlowFragment.Result.PaidByNewCard -> finishWithSuccess(
                paymentId = result.paymentId,
                cardId = result.cardId,
                rebillId = result.rebillId,
            )
        }
    }

    private fun onCardChanged(result: CardListFlowFragment.Result.CardChosen) {
        val card = result.card
        viewModelScope.launch {
            _commandFlow.emit(Command.OpenMainPaymentForm(paymentOptions, card))
        }
    }

    private fun handleMirPayResult(result: MirPaymentResult) {
        when (result) {
            MirPaymentResult.Canceled -> finishWithCancel()
            is MirPaymentResult.Error -> onMirPayError(result.error)
            is MirPaymentResult.Success -> finishWithSuccess(
                paymentId = result.paymentId,
                cardId = null,
                rebillId = null,
            )
        }
    }

    private fun onMirPayError(error: Throwable) {
        returnToMainFormOrFinish(error)
    }

    private fun handleTpayResult(result: TPayResult) {
        when (result) {
            TPayResult.Canceled -> finishWithCancel()
            is TPayResult.Error -> onTpayError(result.error)
            is TPayResult.Success -> finishWithSuccess(
                paymentId = result.paymentId,
                cardId = null,
                rebillId = null,
            )
        }
    }

    private fun onTpayError(error: Throwable) {
        returnToMainFormOrFinish(error)
    }

    private fun returnToMainFormOrFinish(error: Throwable) {
        if (paymentOptions.features.showPaymentNotifications) {
            openMainForm()
        } else {
            finishWithError(error)
        }
    }

    private fun handleSbpResult(result: SbpPaymentFlowFragment.Result) {
        when (result) {
            SbpPaymentFlowFragment.Result.Cancel -> onSbpCancel()
            is SbpPaymentFlowFragment.Result.Error -> onSbpError(result.error)
            is SbpPaymentFlowFragment.Result.Success -> finishWithSuccess(
                paymentId = result.paymentId,
                cardId = null,
                rebillId = null,
            )
        }
    }

    private fun onSbpError(error: Throwable) {
        returnToMainFormOrFinish(error)
    }

    private fun onSbpCancel() {
        openPreviousScreen()
    }

    private fun handlePayByCardResult(result: PaymentByCardFlowFragment.Result) {
        when (result) {
            PaymentByCardFlowFragment.Result.PaymentCancel -> openPreviousScreen()
            is PaymentByCardFlowFragment.Result.PaymentError -> returnToMainFormOrFinish(result.error)
            is PaymentByCardFlowFragment.Result.PaymentSuccess -> finishWithSuccess(
                paymentId = result.paymentId,
                cardId = result.cardId,
                rebillId = result.rebillId,
            )
        }
    }

    private fun handleMainPaymentFormResult(result: MainPaymentFormResult) {
        when (result) {
            MainPaymentFormResult.FinishCancelled -> finishWithCancel()
            is MainPaymentFormResult.FinishError -> returnToMainFormOrFinish(result.error)
            is MainPaymentFormResult.FinishSuccess -> finishWithSuccess(
                paymentId = result.paymentId,
                cardId = result.cardId,
                rebillId = result.rebillId,
            )

            is MainPaymentFormResult.PayByCard -> openPaymentByCard(result)
            is MainPaymentFormResult.PayBySbp -> openPayBySbp(result)
            is MainPaymentFormResult.PayByTpay -> openPayByTpay(result)
            is MainPaymentFormResult.PayByMirPay -> openPayByMirPay(result.paymentOptions)
            is MainPaymentFormResult.ChangeCard -> openCardChange(result.startData)
            is MainPaymentFormResult.NeedThreeDs -> openThreeDs(result)
        }
    }

    private fun openThreeDs(result: MainPaymentFormResult.NeedThreeDs) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.OpenThreeDsFlow(
                    result.options,
                    result.data,
                    result.panSuffix
                )
            )
        }
    }

    private fun openCardChange(startData: ChooseCardLauncher.StartData) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.OpenCardChangeFlow(
                    startData.paymentOptions!!,
                    startData.savedCardsOptions
                )
            )
        }
    }

    private fun openPreviousScreen() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnToMainPaymentForm)
        }
    }

    private fun openPayByMirPay(paymentOptions: PaymentOptions) {
        viewModelScope.launch {
            _commandFlow.emit(Command.OpenPayByPirPayFlow(paymentOptions))
        }
    }

    private fun openPayByTpay(result: MainPaymentFormResult.PayByTpay) {
        viewModelScope.launch {
            _commandFlow.emit(Command.OpenPayByTpayFlow(result))
        }
    }

    private fun openPayBySbp(result: MainPaymentFormResult.PayBySbp) {
        viewModelScope.launch {
            _commandFlow.emit(Command.OpenPayBySbpFlow(result.paymentOptions))
        }
    }

    private fun openPaymentByCard(result: MainPaymentFormResult.PayByCard) {
        viewModelScope.launch {
            _commandFlow.emit(Command.OpenPayByCardFlow(result))
        }
    }

    private fun finishWithSuccess(paymentId: Long, cardId: String?, rebillId: String?) {
        emitResultCommand(
            Result.FinishSuccess(
                paymentId = paymentId,
                cardId = cardId,
                rebillId = rebillId,
            )
        )
    }

    private fun finishWithError(error: Throwable) {
        emitResultCommand(Result.FinishError(error))
    }

    private fun finishWithCancel() {
        emitResultCommand(Result.FinishCancelled)
    }

    private fun emitResultCommand(result: Result) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnResult(result)
            )
        }
    }

    private fun onResume() {
        if (isResumeFirstTime) {
            isResumeFirstTime = false
            onResumeFirstTime()
        }
    }

    private fun onResumeFirstTime() {
        openMainForm()
    }

    private fun openMainForm() {
        viewModelScope.launch {
            _commandFlow.emit(Command.OpenMainPaymentForm(paymentOptions))
        }
    }

    data class State(
        val notification: Notification? = null
    )

    sealed interface Command {
        data object ReturnToMainPaymentForm : Command
        data object OpenEmptyScreen : Command
        class OpenMainPaymentForm(val options: PaymentOptions, val card: Card? = null) : Command
        class OpenPayByCardFlow(val options: MainPaymentFormResult.PayByCard) : Command
        class ReturnResult(val result: Result) : Command
        class OpenPayBySbpFlow(val paymentOptions: PaymentOptions) : Command
        class OpenPayByTpayFlow(val result: MainPaymentFormResult.PayByTpay) : Command
        class OpenPayByPirPayFlow(val paymentOptions: PaymentOptions) : Command
        class OpenCardChangeFlow(
            val paymentOptions: PaymentOptions,
            val savedCardsOptions: SavedCardsOptions
        ) : Command

        class OpenThreeDsFlow(
            val options: PaymentOptions,
            val data: ThreeDsData,
            val panSuffix: String
        ) : Command
    }

    sealed interface Event {
        data object Resume : Event
        class MainPaymentFormResult(
            val result: ru.tinkoff.acquiring.sdk.redesign.mainform.ui.MainPaymentFormResult
        ) : Event

        class PaymentByCardFlowFragmentResult(
            val result: PaymentByCardFlowFragment.Result
        ) : Event

        class SbpPaymentFragmentResult(
            val result: SbpPaymentFlowFragment.Result
        ) : Event

        class TpayPaymentFragmentResult(
            val result: TPayResult
        ) : Event

        class MirPayPaymentFragmentResult(
            val result: MirPaymentResult
        ) : Event

        class CardListFlowFragmentResult(
            val result: CardListFlowFragment.Result
        ) : Event

        class ThreeDsFlowResult(val result: ThreeDsResult) : Event
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data object FinishCancelled : Result

        @Parcelize
        data class FinishSuccess(
            val paymentId: Long,
            val cardId: String? = null,
            val rebillId: String? = null

        ) : Result

        @Parcelize
        data class FinishError(
            val error: Throwable
        ) : Result
    }
}
