package ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.vm

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.mapNotNull
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringNetworkError
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringNetworkException
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.analytics.ChosenMethod
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardProcess
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardState
import ru.tinkoff.acquiring.sdk.redesign.common.result.AcqPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.mainform.navigation.MainFormNavController
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.MainPaymentForm
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.MainPaymentFormFactory
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.analytics.MainFormAnalyticsDelegate
import ru.tinkoff.acquiring.sdk.redesign.payment.PaymentByCardLauncher
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.redesign.tpay.models.getTinkoffPayVersion
import ru.tinkoff.acquiring.sdk.utils.CoroutineManager
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory

/**
 * Created by i.golovachev
 */
internal class MainPaymentFormViewModel(
    private val primaryButtonFactory: MainPaymentFormFactory,
    private val mainFormNavController: MainFormNavController,
    private val paymentByCardProcess: PaymentByCardProcess,
    private val mainFormAnalyticsDelegate: MainFormAnalyticsDelegate,
    private val coroutineManager: CoroutineManager,
    private val paymentOptions: PaymentOptions,
) : ViewModel() {

    private val _formState = MutableStateFlow<MainPaymentForm.State?>(null)
    val primary = _formState.filterNotNull().map { it.ui.primary }
    val secondary = _formState.filterNotNull().map { it.ui.secondaries }
    val chosenCard =
        _formState.mapNotNull { (it?.ui?.primary as? MainPaymentForm.Primary.Card)?.selectedCard }
    val formContent = MutableStateFlow<FormContent>(FormContent.Loading)
    val mainFormNav = mainFormNavController.navFlow

    init {
        loadState()
    }

    fun onRetry() {
        loadState()
    }

    fun toSbp() = viewModelScope.launch(coroutineManager.main) {
        mainFormNavController.toSbp(
            mainFormAnalyticsDelegate.prepareOptions(
                paymentOptions,
                ChosenMethod.Sbp
            )
        )
    }

    fun toPayCard() = viewModelScope.launch(coroutineManager.main) {
        val cards = _formState.value?.data?.cards ?: emptyList()
        val method = if (cards.isEmpty()) ChosenMethod.NewCard else ChosenMethod.Card
        mainFormNavController.toPayCard(
            paymentOptions = mainFormAnalyticsDelegate.prepareOptions(
                paymentOptions,
                method
            ),
            cards = cards,
            chosenCard = _formState.value?.data?.chosen
        )
    }

    fun toChooseCard() = viewModelScope.launch(coroutineManager.main) {
        mainFormNavController.toChooseCard(paymentOptions, _formState.value?.data?.chosen)
    }

    fun toTpay(isPrimary: Boolean = true) = viewModelScope.launch(coroutineManager.main) {
        val version = _formState.value?.data?.info?.getTinkoffPayVersion()
        if (isPrimary && version != null) {
            formContent.value = FormContent.Hide
            mainFormNavController.toTpay(
                paymentOptions = mainFormAnalyticsDelegate.prepareOptions(
                    paymentOptions,
                    ChosenMethod.TinkoffPay
                ),
                version = version
            )
        } else {
            mainFormNavController.toTpayWebView()
        }
    }

    fun toMirPay() = viewModelScope.launch(coroutineManager.main) {
        formContent.value = FormContent.Hide
        val options = mainFormAnalyticsDelegate.prepareOptions(
            paymentOptions,
            ChosenMethod.MirPay
        )
        mainFormNavController.toMirPay(options)
    }

    fun onBackPressed() = returnResult()

    fun returnResult() = viewModelScope.launch {
        val result = when (val it = paymentByCardProcess.state.value) {
            is PaymentByCardState.Error -> PaymentByCardLauncher.Error(
                it.throwable,
                null,
                it.paymentId
            )

            is PaymentByCardState.Success -> PaymentByCardLauncher.Success(
                it.paymentId,
                it.cardId,
                it.rebillId
            )
            PaymentByCardState.Cancelled,
            is PaymentByCardState.Started,
            is PaymentByCardState.Created,
            is PaymentByCardState.ThreeDsInProcess,
            is PaymentByCardState.ThreeDsUiNeeded,
            is PaymentByCardState.CvcUiNeeded,
            is PaymentByCardState.CvcUiInProcess,
            is PaymentByCardState.ThreeDsAppBase ->  PaymentByCardLauncher.Canceled
        }
        mainFormNavController.close(result)
    }

    private fun loadState() {
        coroutineManager.launchOnBackground {
            formContent.value = FormContent.Loading
            runCatching { primaryButtonFactory.getState() }
                .onFailure {
                    onPrimaryButtonGetStateError(it)
                }
                .onSuccess {
                    mainFormAnalyticsDelegate.mapAnalytics(it.ui.primary)
                    _formState.value = it
                    formContent.value = if (it.noInternet) {
                        FormContent.NoNetwork
                    } else {
                        FormContent.Content(it.ui, paymentOptions)
                    }
                }
        }
    }

    private fun onPrimaryButtonGetStateError(e: Throwable) {
        if (paymentOptions.features.showPaymentNotifications) {

                val notification = NotificationFactory.getContentLoadingErrorNotification(e) {
                    when (e) {
                        is AcquiringNetworkError -> ::onBackPressed
                        is AcquiringNetworkException -> ::loadState
                        else -> ::onBackPressed
                    }
                }
            viewModelScope.launch {
                delay(SHIMMER_HIDE_DELAY)
                formContent.value = FormContent.Error(notification)
            }
        } else {
            val errorResult = object : AcqPaymentResult.Error {
                override val error: Throwable = e
                override val errorCode: Int? = null
                override val paymentId: Long? = paymentOptions.paymentId
            }
            viewModelScope.launch {
                mainFormNavController.close(errorResult)
            }
        }
    }

    companion object {
        const val SHIMMER_HIDE_DELAY: Long = 1000
    }

    sealed interface FormContent {

        data object Hide : FormContent

        data object Loading : FormContent

        class Error(val notification: Notification) : FormContent

        data object NoNetwork : FormContent

        class Content(state: MainPaymentForm.Ui, options: PaymentOptions) : FormContent {
            val isSavedCard = (state.primary as? MainPaymentForm.Primary.Card)?.selectedCard != null
            val description: String? = options.order.description
            val showDescription: Boolean = description.isNullOrEmpty().not()
            val isLogoVisible = state.isLogoVisible
        }
    }
}
