package ru.tinkoff.acquiring.sdk.redesign.mainform.ui

import android.graphics.Rect
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import com.google.android.material.bottomsheet.BottomSheetBehavior
import kotlinx.coroutines.android.awaitFrame
import ru.tinkoff.acquiring.sdk.ui.customview.LockableView

/**
 * Created by i.golovachev
 */
internal class BottomSheetComponent(
    private val root: ViewGroup,
    private val sheet: View,
    private val bottomSheetBehavior: BottomSheetBehavior<View> = BottomSheetBehavior.from(sheet),
    private val onSheetHidden: () -> Unit,
    private val onClickOutside: (() -> Boolean)? = null
) {

    private var isLocked: Boolean = false
    private var isHideableStoredValue: Boolean? = null
    private var contentHeight: Int = 0

    init {
        bottomSheetBehavior.addBottomSheetCallback(object :
            BottomSheetBehavior.BottomSheetCallback() {
            override fun onStateChanged(bottomSheet: View, newState: Int) {
                when (newState) {
                    BottomSheetBehavior.STATE_HIDDEN -> onSheetHidden()
                    else -> Unit
                }
            }

            override fun onSlide(bottomSheet: View, slideOffset: Float) = Unit
        })
        setRootOnTouchListener()
    }

    fun collapse() {
        bottomSheetBehavior.state = BottomSheetBehavior.STATE_COLLAPSED
    }

    fun expand() {
        bottomSheetBehavior.state = BottomSheetBehavior.STATE_EXPANDED
    }

    fun hide() {
        bottomSheetBehavior.state = BottomSheetBehavior.STATE_HIDDEN
    }

    fun lock() {
        if (root is LockableView) {
            root.lockView()
        }
        this.isLocked = true
        this.isHideableStoredValue = bottomSheetBehavior.isHideable
        bottomSheetBehavior.isHideable = false
    }

    fun unlock() {
        if (root is LockableView) {
            root.unlockView()
        }
        this.isLocked = false
        this.isHideableStoredValue?.let {
            bottomSheetBehavior.isHideable = it
        }
    }

    fun onAttachedToWindow() {
        expand()
    }

    suspend fun trimSheetToContent(measuredView: View) {
        awaitFrame()
        awaitFrame()
        ViewCompat.getRootWindowInsets(sheet)?.let { insets ->
            val bottom = insets.getInsets(WindowInsetsCompat.Type.navigationBars()).bottom
            val measuredHeight = measuredView.measuredHeight + bottom
            contentHeight = measuredHeight
            bottomSheetBehavior.setPeekHeight(measuredHeight, true)
        }
    }

    private fun setRootOnTouchListener(){
        root.setOnTouchListener { _, event ->
            if (isLocked) return@setOnTouchListener true

            if (event.action == MotionEvent.ACTION_DOWN) {
                val sheetRect = Rect()
                sheet.getGlobalVisibleRect(sheetRect)
                if (!sheetRect.contains(event.rawX.toInt(), event.rawY.toInt())) {
                    onClickOutside?.invoke() ?: false
                }
            }

            false
        }
    }
}
