package ru.tinkoff.acquiring.sdk.redesign.mainform.ui

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.NavHostFragment
import androidx.navigation.navOptions
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.MainPaymentGraphDirections
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqMainPaymentFlowFragmentBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.CardListFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.vm.MainPaymentFormFlowViewModel
import ru.tinkoff.acquiring.sdk.redesign.mirpay.ui.MirPayFragment
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.PaymentByCardFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.sbp.ui.SbpPaymentFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.tpay.ui.TpayFragment
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.toggles.toggles.PaymentByCardV2FeatureToggle
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsFragment
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

/**
 * @author s.y.biryukov
 */
class MainPaymentFormFlowFragment : Fragment(), IsolatedKoinComponent {

    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? = null
    private var viewBinding: AcqMainPaymentFlowFragmentBinding? = null
    private var navHostFragment: NavHostFragment? = null
    private val viewModel: MainPaymentFormFlowViewModel by viewModel<MainPaymentFormFlowViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }
    private val featureToggleManager: FeatureToggleManager by inject()

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val binding = AcqMainPaymentFlowFragmentBinding.inflate(inflater, container, false)
        this.viewBinding = binding
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        this.notificationBottomSheetBinding = NotificationUtils.bindNotificationBottomSheet(
            activity = requireActivity()
        )

        val fragment = childFragmentManager.findFragmentById(R.id.main_content) as NavHostFragment
        navHostFragment = fragment

        registerResultListeners(fragment)

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.CREATED) {
                viewModel.stateFlow.collectLatest {
                    if (it.notification != null) {
                        val data = it.notification.toStatusViewData(requireContext())
                        notificationBottomSheetBinding?.show(data)
                    } else {
                        notificationBottomSheetBinding?.hide()
                    }
                }
            }
        }

        subscribeToCommands()
    }

    private fun subscribeToCommands() {
        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is MainPaymentFormFlowViewModel.Command.OpenMainPaymentForm ->
                            openMainPaymentForm(
                                options = it.options,
                                card = it.card
                            )

                        is MainPaymentFormFlowViewModel.Command.ReturnResult -> setResult(it.result)
                        is MainPaymentFormFlowViewModel.Command.OpenPayByCardFlow ->
                            openPayByCardFlow(
                                it.options
                            )

                        is MainPaymentFormFlowViewModel.Command.OpenPayBySbpFlow -> openPayBySbpFlow(
                            it.paymentOptions
                        )

                        is MainPaymentFormFlowViewModel.Command.OpenPayByTpayFlow -> openPayByTpay(
                            it.result
                        )

                        is MainPaymentFormFlowViewModel.Command.OpenPayByPirPayFlow -> openPayByMirPay(
                            it.paymentOptions
                        )

                        is MainPaymentFormFlowViewModel.Command.OpenCardChangeFlow -> openCardChange(
                            it.paymentOptions, it.savedCardsOptions
                        )

                        is MainPaymentFormFlowViewModel.Command.OpenThreeDsFlow -> openThreeDs(
                            it.options, it.data, it.panSuffix
                        )

                        MainPaymentFormFlowViewModel.Command.ReturnToMainPaymentForm -> {
                            navHostFragment?.navController?.popBackStack(
                                destinationId = R.id.mainPaymentFragment,
                                inclusive = false
                            )
                        }

                        MainPaymentFormFlowViewModel.Command.OpenEmptyScreen -> {
                            navHostFragment?.navController?.navigate(
                                MainPaymentGraphDirections.openEmptyScreen()
                            )
                        }
                    }
                }
            }
        }
    }

    private fun registerResultListeners(fragment: NavHostFragment) {
        val fragmentManager = fragment.childFragmentManager

        MainPaymentFormFragment.registerResultListener(fragmentManager, this) {
            viewModel.handleEvent(MainPaymentFormFlowViewModel.Event.MainPaymentFormResult(it))
        }

        PaymentByCardFlowFragment.registerResultListener(fragmentManager, this) {
            viewModel.handleEvent(
                MainPaymentFormFlowViewModel.Event.PaymentByCardFlowFragmentResult(it)
            )
        }

        SbpPaymentFlowFragment.registerResultListener(fragmentManager, this) {
            viewModel.handleEvent(MainPaymentFormFlowViewModel.Event.SbpPaymentFragmentResult(it))
        }

        TpayFragment.setFragmentResultListener(fragmentManager, this) {
            viewModel.handleEvent(MainPaymentFormFlowViewModel.Event.TpayPaymentFragmentResult(it))
        }

        MirPayFragment.registerFragmentResultListener(fragmentManager, this) {
            viewModel.handleEvent(MainPaymentFormFlowViewModel.Event.MirPayPaymentFragmentResult(it))
        }

        CardListFlowFragment.registerResultListener(fragmentManager, this) {
            viewModel.handleEvent(MainPaymentFormFlowViewModel.Event.CardListFlowFragmentResult(it))
        }

        ThreeDsFragment.registerResultListener(fragmentManager, this) {
            viewModel.handleEvent(MainPaymentFormFlowViewModel.Event.ThreeDsFlowResult(it))
        }
    }

    private fun openThreeDs(options: PaymentOptions, data: ThreeDsData, panSuffix: String) {
        navHostFragment?.navController?.navigate(
            MainPaymentGraphDirections.openThreeDs(options, data, panSuffix)
        )
    }

    private fun openCardChange(
        paymentOptions: PaymentOptions,
        savedCardsOptions: SavedCardsOptions
    ) {
        navHostFragment?.navController?.navigate(
            MainPaymentFormFragmentDirections.changeCard(
                savedCardsOptions, paymentOptions
            )
        )
    }

    private fun openPayByMirPay(paymentOptions: PaymentOptions) {
        navHostFragment?.navController?.navigate(
            MainPaymentFormFragmentDirections.openPayByMirPay(paymentOptions)
        )
    }

    private fun openPayByTpay(result: MainPaymentFormResult.PayByTpay) {
        navHostFragment?.navController?.navigate(
            MainPaymentGraphDirections.openPayByTpay(
                paymentOptions = result.paymentOptions,
                version = result.version
            )
        )
    }

    private fun openPayBySbpFlow(paymentOptions: PaymentOptions) {
        navHostFragment?.navController?.navigate(
            MainPaymentGraphDirections.openPayBySbp(
                paymentOptions = paymentOptions
            )
        )
    }

    private fun openPayByCardFlow(options: MainPaymentFormResult.PayByCard) {
        lifecycleScope.launch {
            navHostFragment?.navController?.navigate(
                MainPaymentGraphDirections.openPaymentByCard(
                    paymentOptions = options.paymentOptions,
                    card = options.cards.firstOrNull(),
                    showArrow = options.withArrowBack,
                    isNewCardViewToggleEnabled = featureToggleManager.isEnabled(PaymentByCardV2FeatureToggle)
                )
            )
        }
    }

    private fun openMainPaymentForm(options: PaymentOptions, card: Card?) {
        navHostFragment?.navController?.navigate(
            MainPaymentGraphDirections.openMainPaymentForm(
                paymentOptions = options,
                card = card
            ),
            navOptions = navOptions {
                launchSingleTop = true
            }
        )
    }

    override fun onResume() {
        super.onResume()

        viewModel.handleEvent(MainPaymentFormFlowViewModel.Event.Resume)
    }

    override fun onDestroyView() {
        super.onDestroyView()
        viewBinding = null
    }

    private fun setResult(result: MainPaymentFormFlowViewModel.Result) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    companion object {
        private const val TAG = "MainPaymentFormFlowFragment"
        private const val FRAGMENT_RESULT_KEY = "$TAG.FRAGMENT_RESULT_KEY"
        private const val FRAGMENT_RESULT_BUNDLE_KEY = "$TAG.FRAGMENT_RESULT_BUNDLE_KEY"
        private const val ARG_MAIN_FORM_OPTION = "paymentOptions"

        fun newInstance(paymentOptions: PaymentOptions): MainPaymentFormFlowFragment {
            return MainPaymentFormFlowFragment().apply {
                arguments = bundleOf(
                    ARG_MAIN_FORM_OPTION to paymentOptions
                )
            }
        }

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (MainPaymentFormFlowViewModel.Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(
                FRAGMENT_RESULT_KEY,
                lifecycleOwner
            ) { _, bundle ->
                val result =
                    bundle.getParcelable<MainPaymentFormFlowViewModel.Result>(
                        FRAGMENT_RESULT_BUNDLE_KEY
                    )
                result?.let(listener)
            }
        }
    }
}

