package ru.tinkoff.acquiring.sdk.redesign.mainform.ui

import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.addCallback
import androidx.core.os.bundleOf
import androidx.core.view.WindowCompat
import androidx.core.view.children
import androidx.core.view.isVisible
import androidx.core.view.updatePadding
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.navArgs
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.take
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject
import org.koin.android.scope.AndroidScopeComponent
import org.koin.androidx.scope.fragmentScope
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import org.koin.core.scope.Scope
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqCardPayComponentBinding
import ru.tinkoff.acquiring.sdk.databinding.AcqMainFormPrimaryButtonComponentBinding
import ru.tinkoff.acquiring.sdk.databinding.AcqMainFormSecondaryBlockBinding
import ru.tinkoff.acquiring.sdk.databinding.FragmentMainPaymentFormBinding
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.result.PaymentResult
import ru.tinkoff.acquiring.sdk.redesign.common.cardpay.CardPayComponent
import ru.tinkoff.acquiring.sdk.redesign.common.result.AcqPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.common.util.AcqShimmerAnimator
import ru.tinkoff.acquiring.sdk.redesign.dialog.PaymentStatusSheetState
import ru.tinkoff.acquiring.sdk.redesign.dialog.toStatusType
import ru.tinkoff.acquiring.sdk.redesign.mainform.navigation.MainFormNavController
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.MainPaymentFormFactory
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.vm.MainFormInputCardViewModel
import ru.tinkoff.acquiring.sdk.redesign.mainform.presentation.vm.MainPaymentFormViewModel
import ru.tinkoff.acquiring.sdk.smartfield.AcqTextFieldView
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsLauncher
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.keyboard.SecureKeyboardController
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegate
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegateImpl
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory
import ru.tinkoff.acquiring.sdk.utils.applyThemeMode
import ru.tinkoff.acquiring.sdk.utils.getString
import ru.tinkoff.acquiring.sdk.utils.setupCvcInput
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData
import java.lang.ref.WeakReference

class MainPaymentFormFragment : Fragment(),
    AppBaseChallengeDelegate by AppBaseChallengeDelegateImpl(), AndroidScopeComponent,
    IsolatedKoinComponent {

    override val scope: Scope by fragmentScope(true)

    private val args: MainPaymentFormFragmentArgs by navArgs()

    private val options: PaymentOptions get() = args.paymentOptions

    private var viewBinding: FragmentMainPaymentFormBinding? = null
    private var emailJob: Job? = null

    private val mainPaymentFormFactory by inject<MainPaymentFormFactory> {
        parametersOf(args.card, args.paymentOptions)
    }

    private val viewModel: MainPaymentFormViewModel by viewModel {
        parametersOf(args.paymentOptions, mainPaymentFormFactory)
    }

    private val cardInputViewModel: MainFormInputCardViewModel by viewModel {
        parametersOf(
            SavedStateViewModelFactory(
                requireActivity().application, this, arguments
            ),
            args.paymentOptions
        )
    }

    private var bottomSheetComponent: BottomSheetComponent? = null
    private var primaryButtonComponent: PrimaryButtonComponent? = null
    private var cardPayComponent: CardPayComponent? = null
    private var secondaryButtonComponent: SecondaryBlockComponent? = null
    private var keyboardStateListener: SecureKeyboardController.KeyboardStateListener? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return FragmentMainPaymentFormBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        val activityScope = (activity as? AndroidScopeComponent)?.scope
        if (activityScope != null) {
            scope.linkTo(activityScope)
        }
        val viewBinding = this.viewBinding ?: return

        val bottomSheetComponent =
            BottomSheetComponent(
                root = viewBinding.root,
                sheet = viewBinding.acqMainFormSheet,
                onSheetHidden = {
                    viewModel.onBackPressed()
                },
                onClickOutside = {
                    this.bottomSheetComponent?.hide()
                    true
                }
            )
        this.bottomSheetComponent = bottomSheetComponent

        val primaryButtonComponent = PrimaryButtonComponent(
            viewBinding = AcqMainFormPrimaryButtonComponentBinding.bind(
                requireView().findViewById(R.id.acq_main_form_primary_button)
            ),
            onMirPayClick = viewModel::toMirPay,
            onNewCardClick = viewModel::toPayCard,
            onSpbClick = viewModel::toSbp,
            onTpayClick = viewModel::toTpay,
            onPayClick = cardInputViewModel::pay
        )
        this.primaryButtonComponent = primaryButtonComponent

        val currentClassWR = WeakReference(this)
        val bottomSheetComponentWR = WeakReference(bottomSheetComponent)
        val cardPayComponent = CardPayComponent(
            WeakReference(viewBinding.root),
            viewBinding = AcqCardPayComponentBinding.bind(
                requireView().findViewById(R.id.acq_main_card_pay)
            ),
            email = options.customer.email,
            onCvcCompleted = cardInputViewModel::setCvc,
            onEmailInput = cardInputViewModel::email,
            onEmailVisibleChange = {
                emailJob = currentClassWR.get()?.lifecycleScope?.launch {
                    bottomSheetComponentWR.get()?.trimSheetToContent(viewBinding.acqMainFormContent)
                }
                cardInputViewModel.needEmail(it)
            },
            onChooseCardClick = viewModel::toChooseCard,
            onPayClick = { cardInputViewModel.pay() },
            useSecureKeyboard = options.features.useSecureKeyboard
        )
        this.cardPayComponent = cardPayComponent

        val secondaryButtonComponent = SecondaryBlockComponent(
            binding = AcqMainFormSecondaryBlockBinding.bind(
                requireView().findViewById(R.id.acq_main_form_secondary_button)
            ),
            onNewCardClick = viewModel::toPayCard,
            onSpbClick = viewModel::toSbp,
            onTpayClick = { viewModel.toTpay(false) },
            onMirPayClick = viewModel::toMirPay,
        )
        this.secondaryButtonComponent = secondaryButtonComponent

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.onBackPressed()
        }

        viewBinding.root.post {
            bottomSheetComponent.onAttachedToWindow()
        }

        initCvcClickListener(viewBinding.root, cardPayComponent)
        initTheme()
        createTitleView()
        lifecycleScope.launch { updateContent() }
        lifecycleScope.launch { updatePayEnable() }
        lifecycleScope.launch { updateButtonLoader() }
        lifecycleScope.launch { updatePrimary() }
        lifecycleScope.launch { updateSecondary() }
        lifecycleScope.launch { updateSavedCard() }
        lifecycleScope.launch { updateCardPayState() }
        lifecycleScope.launch { updateLoadingState() }
        lifecycleScope.launch {
            cardInputViewModel.savedCardFlow.collectLatest {
                cardPayComponent.renderNewCard(it)
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            lifecycle.repeatOnLifecycle(Lifecycle.State.STARTED) {
                subscribeOnNav()
            }
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        emailJob?.cancel()
        emailJob = null
        viewBinding = null
        bottomSheetComponent = null
        primaryButtonComponent = null
        cardPayComponent?.clear()
        cardPayComponent = null
        secondaryButtonComponent = null
    }

    override fun onResume() {
        super.onResume()
        val secureKeyboardController = SecureKeyboardController.getInstance()
        viewBinding?.let { secureKeyboardController.registerInsets(it.root) }
        val listener = object :
            SecureKeyboardController.KeyboardStateListener {
            override fun onPaddingUpdated(height: Int, navigationHeight: Int): Boolean {
                viewBinding?.acqMainFormFlipper?.updatePadding(
                    bottom = maxOf(
                        height,
                        navigationHeight
                    )
                )
                if (height > navigationHeight) {
                    bottomSheetComponent?.expand()
                }
                return true
            }
        }
        this.keyboardStateListener = listener
        secureKeyboardController.setKeyboardStateListener(listener)
    }

    override fun onPause() {
        super.onPause()
        this.keyboardStateListener = null
        SecureKeyboardController.getInstance().clear()
    }

    private fun initCvcClickListener(root: ViewGroup, card: CardPayComponent) {
        root.findViewById<AcqTextFieldView>(R.id.cvc_input).setupCvcInput(root, card)
    }

    private suspend fun updateLoadingState() {
        cardInputViewModel.loadingState.collectLatest {
            if (it != null && options.features.showPaymentNotifications) {
                val binding = viewBinding ?: return@collectLatest
                binding.acqMainFormLoader.root.isVisible = false
                binding.acqMainFormContent.isVisible = false
                binding.acqMainFormStatusView.apply {
                    isVisible = true
                    showNotification(it.toStatusViewData(requireContext()))
                }
            } else {
                bottomSheetComponent?.collapse()
            }
        }
    }

    private suspend fun updateContent() {
        combine(
            cardInputViewModel.paymentStatus, viewModel.formContent
        ) { cardStatus, formContent ->
            val viewBinding = viewBinding ?: return@combine
            if (cardStatus is PaymentStatusSheetState.Error ||
                cardStatus is PaymentStatusSheetState.Success
            ) {
                if (options.features.showPaymentNotifications) {
                    viewBinding.acqMainFormLoader.root.isVisible = false
                    viewBinding.acqMainFormContent.isVisible = false
                    viewBinding.acqMainFormStatusView.apply {
                        isVisible = true
                        val data = StatusViewData(
                            type = cardStatus.toStatusType(),
                            title = getString(cardStatus.title),
                            description = getString(cardStatus.subtitle),
                            buttonText = getString(cardStatus.button),
                            amount = getAmountStringOnlyForSuccessStatus(cardStatus),
                            onButtonClick = {
                                viewModel.onBackPressed()
                            },
                            onCloseByUser = { viewModel.onBackPressed() }
                        )
                        showNotification(data)
                    }
                    bottomSheetComponent?.collapse()
                } else {
                    bottomSheetComponent?.collapse()
                    viewModel.returnResult()
                }
            } else {
                when (formContent) {
                    is MainPaymentFormViewModel.FormContent.Loading -> {
                        viewBinding.acqMainFormStatusView.isVisible = false
                        viewBinding.acqMainFormLoader.root.isVisible = true
                        viewBinding.acqMainFormContent.isVisible = false
                        bottomSheetComponent?.trimSheetToContent(viewBinding.acqMainFormLoader.root)
                        AcqShimmerAnimator.animateSequentially(viewBinding.acqMainFormLoader.root.children.toList())
                    }

                    is MainPaymentFormViewModel.FormContent.Error -> {
                        viewBinding.acqMainFormLoader.root.isVisible = false
                        viewBinding.acqMainFormContent.isVisible = false
                        viewBinding.acqMainFormStatusView.apply {
                            isVisible = true
                            val data = formContent.notification.toStatusViewData(requireContext())
                            showNotification(data)
                            bottomSheetComponent?.trimSheetToContent(this)
                        }
                        bottomSheetComponent?.collapse()
                    }

                    is MainPaymentFormViewModel.FormContent.Content -> {
                        viewBinding.acqMainFormLoader.root.isVisible = false
                        viewBinding.acqMainFormContent.isVisible = true
                        viewBinding.acqMainFormStatusView.isVisible = false
                        cardPayComponent?.isVisible(formContent.isSavedCard)
                        primaryButtonComponent?.isVisible(formContent.isSavedCard.not())
                        viewBinding.acqMainFormDescription.text = formContent.description
                        viewBinding.acqMainFormDescription.isVisible = formContent.showDescription
                        viewBinding.acqIcMainFormGerb.alpha =
                            if (formContent.isLogoVisible) 1f else .0f
                        bottomSheetComponent?.trimSheetToContent(viewBinding.acqMainFormContent)
                        bottomSheetComponent?.collapse()
                    }

                    is MainPaymentFormViewModel.FormContent.NoNetwork -> {
                        viewBinding.acqMainFormLoader.root.isVisible = false
                        viewBinding.acqMainFormContent.isVisible = false
                        viewBinding.acqMainFormStatusView.apply {
                            isVisible = true
                            val notification = NotificationFactory.getOfflineNotification(
                                onClickFactory = { viewModel::onRetry }
                            )
                            showNotification(notification.toStatusViewData(requireContext()))
                            bottomSheetComponent?.trimSheetToContent(this)
                        }
                        bottomSheetComponent?.collapse()
                    }

                    is MainPaymentFormViewModel.FormContent.Hide -> {
                        viewBinding.acqMainFormLoader.root.isVisible = false
                        viewBinding.acqMainFormContent.isVisible = false
                        viewBinding.acqMainFormStatusView.isVisible = false
                        bottomSheetComponent?.collapse()
                    }
                }
            }
        }.collect()
    }

    private fun getAmountStringOnlyForSuccessStatus(cardStatus: PaymentStatusSheetState?) =
        (cardStatus as? PaymentStatusSheetState.Success)?.amount?.toHumanReadableString()
            ?.let { "-$it" }

    private suspend fun updatePrimary() = viewModel.primary.collect {
        primaryButtonComponent?.render(it)
    }

    private suspend fun updateSecondary() = viewModel.secondary.collect {
        secondaryButtonComponent?.render(it)
    }

    private suspend fun updateSavedCard() = viewModel.chosenCard.collect {
        cardInputViewModel.choseCard(it)
    }

    private suspend fun updatePayEnable() = cardInputViewModel.payEnable.collectLatest {
        cardPayComponent?.renderEnable(it)
    }

    private suspend fun updateButtonLoader() =
        cardInputViewModel.isLoading.collectLatest { isLoading ->
            cardPayComponent?.renderLoader(isLoading)
            if (isLoading) {
                bottomSheetComponent?.lock()
                cardPayComponent?.isKeyboardVisible(false)
            } else {
                bottomSheetComponent?.unlock()
            }
            handleLoadingInProcess(isLoading)
        }

    private suspend fun updateCardPayState() = with(cardInputViewModel) {
        combine(savedCardFlow, emailFlow) { card, email -> card to email }
            .take(1)
            .collectLatest { (card, email) ->
                cardPayComponent?.render(card, email, options)
            }
    }

    private suspend fun subscribeOnNav() {
        viewModel.mainFormNav.collect {
            when (it) {
                is MainFormNavController.Navigation.ToChooseCard -> {
                    cardPayComponent?.isKeyboardVisible(false)
                    setResult(MainPaymentFormResult.ChangeCard(it.startData))
                }

                is MainFormNavController.Navigation.ToPayByCard -> {
                    setResult(
                        MainPaymentFormResult.PayByCard(
                            paymentOptions = it.startData.paymentOptions,
                            cards = it.startData.cards,
                            withArrowBack = it.startData.withArrowBack
                        )
                    )
                }

                is MainFormNavController.Navigation.ToSbp -> {
                    setResult(MainPaymentFormResult.PayBySbp(it.startData.paymentOptions))
                }

                is MainFormNavController.Navigation.ToTpay -> {
                    setResult(
                        MainPaymentFormResult.PayByTpay(
                            it.startData.paymentOptions,
                            it.startData.version
                        )
                    )
                }

                is MainFormNavController.Navigation.ToMirPay -> {
                    setResult(MainPaymentFormResult.PayByMirPay(it.startData.paymentOptions))
                }

                is MainFormNavController.Navigation.To3ds -> {
                    setResult(
                        MainPaymentFormResult.NeedThreeDs(
                            options = it.paymentOptions,
                            data = it.threeDsState.data,
                            panSuffix = "",
                        )
                    )
                }

                is MainFormNavController.Navigation.Return -> {
                    when (it.result) {
                        is AcqPaymentResult.Canceled -> setResult(MainPaymentFormResult.FinishCancelled)
                        is AcqPaymentResult.Error -> {
                            setResult(
                                MainPaymentFormResult.FinishError(
                                    it.result.error
                                )
                            )
                        }

                        is AcqPaymentResult.Success -> {
                            setResult(
                                MainPaymentFormResult.FinishSuccess(
                                    paymentId = it.result.paymentId,
                                    cardId = it.result.cardId,
                                    rebillId = it.result.rebillId,
                                )
                            )
                        }
                    }
                }

                is MainFormNavController.Navigation.ToWebView -> {
                    startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(it.url)))
                }

                null -> Unit
                is MainFormNavController.Navigation.ToAppBaseChallenge -> {
                    initAppBaseChallengeDelegate(
                        requireActivity(),
                        AcquiringSdk(
                            options.terminalKey,
                            options.publicKey,
                        ),
                        viewLifecycleOwner.lifecycleScope
                    )

                    launchChallenge(
                        it.threedsData,
                        it.transaction,
                        cardInputViewModel::processChallengeResult
                    )
                }
            }
        }
    }

    private fun createTitleView() {
        viewBinding?.acqMainFormAmount?.text = options.order.amount.toHumanReadableString()
    }

    private fun handleLoadingInProcess(inProcess: Boolean) {
        cardPayComponent?.isEnable(inProcess.not())
    }

    private fun initTheme() {
        applyThemeMode(options.features.darkThemeMode)
    }

    private fun setResult(result: MainPaymentFormResult) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    private fun on3DsError(it: ThreeDsLauncher.Result.Error) {
        cardInputViewModel.set3dsResult(it)
    }

    private fun on3DsError(it: Throwable, paymentId: Long) {
        cardInputViewModel.set3dsResult(it, paymentId)
    }

    private fun on3DsSuccess(it: ThreeDsLauncher.Result.Success) {
        cardInputViewModel.set3dsResult(it.result as PaymentResult)
    }

    private fun on3DsSuccess(paymentResult: PaymentResult) {
        cardInputViewModel.set3dsResult(paymentResult)
    }

    internal companion object {

        private const val TAG = "MainPaymentFormFragment"
        private const val FRAGMENT_RESULT_KEY = "$TAG.FRAGMENT_RESULT_KEY"
        private const val FRAGMENT_RESULT_BUNDLE_KEY = "$TAG.FRAGMENT_RESULT_BUNDLE_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (MainPaymentFormResult) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(
                FRAGMENT_RESULT_KEY,
                lifecycleOwner
            ) { _, bundle ->
                val result =
                    bundle.getParcelable<MainPaymentFormResult>(FRAGMENT_RESULT_BUNDLE_KEY)
                result?.let(listener)
            }
        }
    }
}
