package ru.tinkoff.acquiring.sdk.redesign.mirpay.presentation

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.payment.MirPayPaymentState
import ru.tinkoff.acquiring.sdk.payment.MirPayProcess
import ru.tinkoff.acquiring.sdk.redesign.mirpay.ui.MirPayFragmentArgs
import ru.tinkoff.acquiring.sdk.redesign.mirpay.ui.MirPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory

/**
 * @author k.shpakovskiy
 */
internal class MirPayViewModel(
    savedStateHandle: SavedStateHandle,
    private val mirPayProcess: MirPayProcess
) : ViewModel() {
    private val args = MirPayFragmentArgs.fromSavedStateHandle(savedStateHandle)

    private val paymentOptions: PaymentOptions get() = args.paymentOptions
    private var isFirstResume: Boolean = true

    private val _stateFlow = MutableStateFlow(State())
    val stateFlow = _stateFlow.asStateFlow()

    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()

    init {
        mirPayProcess.stateFlow
            .onEach(::handlePaymentProcessState)
            .launchIn(viewModelScope)
    }

    private fun handlePaymentProcessState(state: MirPayPaymentState) {
        when (state) {
            is MirPayPaymentState.Created -> {
                onPaymentCreated()
            }

            is MirPayPaymentState.NeedChooseOnUi -> {
                viewModelScope.launch {
                    _commandFlow.emit(
                        Command.GoToMirPay(state.deeplink)
                    )
                }
            }

            is MirPayPaymentState.Success -> {
                onPaymentSuccess(state)
            }

            is MirPayPaymentState.PaymentFailed -> {
                onPaymentFailed(state)
            }
            is MirPayPaymentState.Paused,
            is MirPayPaymentState.Stopped,
            is MirPayPaymentState.CheckingStatus,
            is MirPayPaymentState.LeaveOnBankApp,
            is MirPayPaymentState.Started -> Unit
        }
    }

    private fun onPaymentFailed(state: MirPayPaymentState.PaymentFailed) {
        val onClick: () -> Unit = {
            closeWithError(state)
        }
        if (paymentOptions.features.showPaymentNotifications) {
            val notification = NotificationFactory.getPaymentErrorNotification(
                exception = state.throwable,
                onClickFactory = { onClick },
                options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
            )
            _stateFlow.update {
                it.copy(notification = notification)
            }
        } else {
            onClick()
        }
    }

    private fun onPaymentSuccess(state: MirPayPaymentState.Success) {
        val onClick: () -> Unit = {
            closeWithSuccess(state)
        }
        if (paymentOptions.features.showPaymentNotifications) {
            val notification = NotificationFactory.getPaymentSuccessCommonNotification(
                onClick = onClick,
                amount = paymentOptions.order.amount
            )
            _stateFlow.update {
                it.copy(notification = notification)
            }
        } else {
            onClick()
        }
    }

    private fun onPaymentCreated() {
        _stateFlow.update {
            val notification = Notification(
                type = StatusViewData.Type.PROGRESS,
                title = R.string.acq_commonsheet_mir_pay_payment_waiting_title,
                description = R.string.acq_commonsheet_mir_pay_payment_waiting_sub_title,
                button = R.string.acq_commonsheet_mir_pay_payment_waiting_close_button,
                onClick = {
                    closeWithCancelResult()
                }
            )
            it.copy(notification = notification)
        }
    }

    private fun closeWithError(state: MirPayPaymentState.PaymentFailed) {
        val error = state.throwable
        val paymentId = state.paymentId
        val errorCode = state.errorCode
        closeWithError(error, paymentId, errorCode)
    }

    private fun closeWithError(error: Throwable, paymentId: Long?, errorCode: String?) {
        viewModelScope.launch {
            _stateFlow.update {
                it.copy(notification = null)
            }
            _commandFlow.emit(
                Command.CloseWithResult(
                    MirPaymentResult.Error(
                        error = error,
                        paymentId = paymentId,
                        errorCode = errorCode,
                    )
                )
            )
        }
    }

    private fun closeWithSuccess(state: MirPayPaymentState.Success) {
        viewModelScope.launch {
            _stateFlow.update {
                it.copy(notification = null)
            }
            _commandFlow.emit(
                Command.CloseWithResult(
                    MirPaymentResult.Success(
                        paymentId = state.paymentId,
                    )
                )
            )
        }
    }

    private fun closeWithCancelResult() {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.CloseWithResult(
                    MirPaymentResult.Canceled
                )
            )
        }
    }

    private fun pay() {
        mirPayProcess.start(paymentOptions = paymentOptions)
    }

    private fun startCheckingStatus() {
        mirPayProcess.startCheckingStatus()
    }

    fun handleEvent(event: Event) {
        when (event) {
            Event.BackPressed -> onBackPressed()
            Event.Resume -> onResume()
            Event.Pause -> onPause()
            Event.BottomSheetClose -> onBottomSheetClose()
            Event.MirPayOpened -> onOpenMirPaySuccess()
            is Event.CantOpenMirPay -> onOpenMirPayError(event.error)
        }
    }

    private fun onOpenMirPaySuccess() {
        mirPayProcess.goingToBankApp()
    }

    private fun onOpenMirPayError(error: Throwable) {
        val onClose = {
            closeWithError(
                error = error,
                paymentId = paymentOptions.paymentId,
                errorCode = null
            )
        }

        val notification = NotificationFactory.getPaymentErrorNotification(
            exception = error,
            onClickFactory = { onClose },
            options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
        )

        _stateFlow.update {
            it.copy(notification = notification)
        }
    }

    private fun onBottomSheetClose() {
        closeWithCurrentResult()
    }

    private fun onResume() {
        if (isFirstResume) {
            isFirstResume = false
            pay()
        }
        startCheckingStatus()
    }

    private fun onPause() {
        mirPayProcess.pause()
    }

    private fun onBackPressed() {
        closeWithCurrentResult()
    }

    private fun closeWithCurrentResult() {
        when (val state = mirPayProcess.stateFlow.value) {
            is MirPayPaymentState.Success -> closeWithSuccess(state)
            is MirPayPaymentState.PaymentFailed -> closeWithError(state)
            else -> closeWithCancelResult()
        }
    }

    override fun onCleared() {
        mirPayProcess.stop()
        super.onCleared()
    }

    companion object {
        const val TAG = "MirPayViewModel"
    }

    data class State(
        val notification: Notification? = null
    )

    sealed interface Command {
        class CloseWithResult(
            val result: MirPaymentResult
        ) : Command

        class GoToMirPay(
            val deeplink: String
        ) : Command
    }

    sealed interface Event {
        data object BackPressed : Event
        data object Resume : Event
        data object Pause : Event
        data object BottomSheetClose : Event
        data object MirPayOpened : Event
        class CantOpenMirPay(val error: Throwable) : Event
    }
}
