package ru.tinkoff.acquiring.sdk.redesign.mirpay.ui

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.addCallback
import androidx.core.os.bundleOf
import androidx.core.view.WindowCompat
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.viewModels
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.createSavedStateHandle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.lifecycle.viewmodel.initializer
import androidx.lifecycle.viewmodel.viewModelFactory
import androidx.navigation.fragment.navArgs
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.databinding.FragmentMirPayBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.payment.MirPayProcess
import ru.tinkoff.acquiring.sdk.redesign.common.util.DeepLinkOpener
import ru.tinkoff.acquiring.sdk.redesign.mainform.ui.BottomSheetComponent
import ru.tinkoff.acquiring.sdk.redesign.mirpay.presentation.MirPayViewModel
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

class MirPayFragment : Fragment(), IsolatedKoinComponent {

    private var viewBinding: FragmentMirPayBinding? = null
    private var bottomSheetComponent: BottomSheetComponent? = null
    private val args: MirPayFragmentArgs by navArgs()

    private val viewModel: MirPayViewModel by viewModels {
        viewModelFactory {
            initializer {
                val savedStateHandle = createSavedStateHandle()
                val paymentOptions = args.paymentOptions
                val sdk = AcquiringSdk(
                    terminalKey = paymentOptions.terminalKey,
                    publicKey = paymentOptions.publicKey,
                )
                MirPayViewModel(
                    savedStateHandle,
                    MirPayProcess(sdk)
                )
            }
        }
    }

    private val deepLinkOpener by inject<DeepLinkOpener> { parametersOf(requireActivity()) }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return FragmentMirPayBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        bindView()

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.handleEvent(MirPayViewModel.Event.BackPressed)
        }

        updateSheetState()
        subscribeOnCommands()
    }

    override fun onResume() {
        super.onResume()
        viewModel.handleEvent(MirPayViewModel.Event.Resume)
    }

    override fun onPause() {
        super.onPause()
        viewModel.handleEvent(MirPayViewModel.Event.Pause)
    }

    private fun bindView() {

        viewBinding?.let {
            bottomSheetComponent = BottomSheetComponent(
                root = it.root,
                sheet = it.acqMirPayFormSheet,
                onSheetHidden = {
                    viewModel.handleEvent(MirPayViewModel.Event.BottomSheetClose)
                },
                onClickOutside = {
                    bottomSheetComponent?.hide()
                    true
                }
            )
        }
    }

    private fun updateSheetState() = lifecycleScope.launch {
        viewModel.stateFlow.collectLatest { state ->
            showNotification(state.notification)

        }
    }

    private suspend fun showNotification(notification: Notification?) {
        if (notification == null) return
        val viewBinding = viewBinding ?: return

        val data = notification.toStatusViewData(requireContext())
        viewBinding.acqMirPayStatusView.showNotification(data)
        bottomSheetComponent?.trimSheetToContent(viewBinding.acqMirPayFormSheet)
    }

    private fun subscribeOnCommands() = lifecycleScope.launch {
        repeatOnLifecycle(Lifecycle.State.STARTED) {
            viewModel.commandFlow.collect {
                when (it) {
                    is MirPayViewModel.Command.CloseWithResult -> {
                        setResult(it.result)
                    }

                    is MirPayViewModel.Command.GoToMirPay -> {
                        try {
                            deepLinkOpener.openDeepLink(it.deeplink, MIR_PAY_REQUEST_CODE)
                            viewModel.handleEvent(MirPayViewModel.Event.MirPayOpened)
                        } catch (e: Throwable) {
                            viewModel.handleEvent(MirPayViewModel.Event.CantOpenMirPay(e))
                        }
                    }
                }
            }
        }
    }

    private fun setResult(result: MirPaymentResult) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    override fun onDestroyView() {
        super.onDestroyView()
        viewBinding = null
    }

    companion object {

        const val MIR_PAY_REQUEST_CODE = 105

        fun newInstance(paymentOptions: PaymentOptions): MirPayFragment {
            return MirPayFragment().apply {
                arguments = MirPayFragmentArgs(paymentOptions).toBundle()
            }
        }

        fun registerFragmentResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (MirPaymentResult) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(
                FRAGMENT_RESULT_KEY,
                lifecycleOwner
            ) { _, bundle ->
                val result =
                    bundle.getParcelable<MirPaymentResult>(FRAGMENT_RESULT_BUNDLE_KEY)
                result?.let(listener)
            }
        }

        private const val TAG = "MirPayFragment"
        private const val FRAGMENT_RESULT_KEY = "$TAG.FRAGMENT_RESULT_KEY"
        private const val FRAGMENT_RESULT_BUNDLE_KEY = "$TAG.FRAGMENT_RESULT_BUNDLE_KEY"
    }
}
