package ru.tinkoff.acquiring.sdk.redesign.payment

import android.content.Context
import android.content.Intent
import android.os.Parcelable
import androidx.activity.result.contract.ActivityResultContract
import kotlinx.parcelize.Parcelize
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.enums.CardStatus
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.common.result.AcqPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.main.SingleActivity
import ru.tinkoff.acquiring.sdk.redesign.main.SingleActivity.Result.PaymentByCardResult

object PaymentByCardLauncher {

    sealed class Result
    class Success(
        override val paymentId: Long,
        override val cardId: String? = null,
        override val rebillId: String? = null
    ) : Result(), AcqPaymentResult.Success

    object Canceled : Result(), AcqPaymentResult.Canceled
    class Error(
        override val error: Throwable,
        override val errorCode: Int?,
        override val paymentId: Long?
    ) : Result(), AcqPaymentResult.Error

    @Parcelize
    class StartData(
        val paymentOptions: PaymentOptions,
        val cards: List<Card>,
        val withArrowBack: Boolean = false
    ) : Parcelable

    object Contract : ActivityResultContract<StartData, Result>() {

        override fun createIntent(context: Context, input: StartData): Intent {
            return SingleActivity.createIntent(
                context, SingleActivity.Arguments.PaymentByCardArguments(
                    options = input.paymentOptions,
                    card = getFirstActiveCard(input.cards),
                    showArrow = input.withArrowBack
                )
            )
        }

        override fun parseResult(resultCode: Int, intent: Intent?): Result {
            return when (val result: PaymentByCardResult? = SingleActivity.getResult(intent)) {
                is PaymentByCardResult.Success -> {
                    Success(
                        paymentId = result.paymentId,
                        cardId = result.cardId,
                        rebillId = result.rebillId,
                    )
                }

                is PaymentByCardResult.Error -> {
                    Error(
                        error = result.error,
                        errorCode = result.errorCode,
                        paymentId = result.paymentId
                    )
                }

                else -> Canceled
            }
        }

        private fun getFirstActiveCard(cards: List<Card>) =
            cards.find { it.status == CardStatus.ACTIVE }

    }
}
