package ru.tinkoff.acquiring.sdk.redesign.payment.ui

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.setFragmentResult
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.NavDirections
import androidx.navigation.NavOptions
import androidx.navigation.fragment.NavHostFragment
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.PaymentByCardGraphDirections
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqPaymentByCardFlowFragmentBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.models.result.PaymentResult
import ru.tinkoff.acquiring.sdk.redesign.cards.list.ui.CardListFlowFragment
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.v2.PaymentByNewCardFragment
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.v2.PaymentBySavedCardFragment
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsFragment
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsResult
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

/**
 * @author s.y.biryukov
 */
class PaymentByCardFlowFragment : Fragment(), IsolatedKoinComponent {
    private var viewBinding: AcqPaymentByCardFlowFragmentBinding? = null
    private var navHostFragment: NavHostFragment? = null
    private val viewModel by viewModel<PaymentByCardFlowViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }
    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? =
        null

    private val isNewCardViewToggleEnabled: Boolean get() = arguments?.getBoolean("isNewCardViewToggleEnabled", false) == true

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val binding = AcqPaymentByCardFlowFragmentBinding.inflate(layoutInflater, container, false)
        viewBinding = binding
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val fragment = childFragmentManager.findFragmentById(R.id.main_content) as NavHostFragment
        navHostFragment = fragment

        registerFragmentResults(fragment)

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is PaymentByCardFlowViewModel.Command.ReturnPaymentSuccessResult -> {
                            returnResult(
                                Result.PaymentSuccess(
                                    paymentId = it.paymentId,
                                    rebillId = it.rebillId,
                                    cardId = it.cardId
                                )
                            )
                        }

                        is PaymentByCardFlowViewModel.Command.ReturnPaymentErrorResult -> {
                            returnResult(Result.PaymentError(
                                error = it.error,
                                paymentId = it.paymentId
                            ))
                        }

                        PaymentByCardFlowViewModel.Command.ReturnPaymentCancelResult -> {
                            returnResult(Result.PaymentCancel)
                        }

                        is PaymentByCardFlowViewModel.Command.ChangeCard -> {
                            changeCard(
                                savedCardsOptions = it.savedCardsOptions,
                                paymentOptions = it.paymentOptions,
                                card = it.card
                            )
                        }

                        PaymentByCardFlowViewModel.Command.ReturnToPrevScreen -> {
                            navHostFragment?.navController?.popBackStack()
                        }

                        is PaymentByCardFlowViewModel.Command.CheckThreeDs -> {
                            navHostFragment?.navController?.navigate(
                                PaymentByCardGraphDirections.openThreeDs(
                                    threeDsData = it.data,
                                    panSuffix = "",
                                    options = it.paymentOptions,
                                )
                            )
                        }

                        is PaymentByCardFlowViewModel.Command.PayWithNewCard -> {
                            val direction = getPayByCardDirection(it.paymentOptions, it.card)
                            navHostFragment?.navController?.navigate(
                                directions = direction,
                                navOptions = NavOptions.Builder()
                                    .setLaunchSingleTop(true)
                                    .build()
                            )
                        }

                        PaymentByCardFlowViewModel.Command.ReturnToEmptyScreen -> {
                            navHostFragment?.navController?.popBackStack(
                                R.id.emptyFragment,
                                false
                            )
                        }
                    }
                }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.state.collectLatest {
                    val notification = it.notification
                    if (notification != null) {
                        val data = notification.toStatusViewData(requireContext())
                        notificationBottomSheetBinding?.show(data)
                    } else {
                        notificationBottomSheetBinding?.hide()
                    }
                }
            }
        }

        this.notificationBottomSheetBinding = NotificationUtils.bindNotificationBottomSheet(
            activity = requireActivity()
        )

        if (savedInstanceState == null) {
            viewLifecycleOwner.lifecycleScope.launch {
                openPaymentByCard()
            }
        }
    }

    private suspend fun getPayByCardDirection(paymentOptions: PaymentOptions, card: Card?, showArrow: Boolean = false): NavDirections {
        val direction = if (isNewCardViewToggleEnabled) {
            if (card == null) {
                PaymentByCardGraphDirections.openPaymentByNewCard(
                    paymentOptions = paymentOptions,
                    showArrow = showArrow,
                )
            } else {
                PaymentByCardGraphDirections.openPaymentBySavedCard(
                    paymentOptions = paymentOptions,
                    card = card,
                    showArrow = showArrow,
                )
            }
        } else {
            PaymentByCardGraphDirections.openPaymentByCard(
                paymentOptions = paymentOptions,
                card = card,
                showArrow = showArrow,
            )
        }
        return direction
    }

    private fun changeCard(
        savedCardsOptions: SavedCardsOptions,
        paymentOptions: PaymentOptions,
        card: Card?
    ) {
        navHostFragment?.navController?.navigate(
            PaymentByCardGraphDirections.changeCard(
                savedCardsOptions = savedCardsOptions,
                paymentOptions = paymentOptions,
                card = card
            )
        )
    }

    private fun registerFragmentResults(fragment: NavHostFragment) {
        PaymentBySavedCardFragment.registerResultListener(fragment.childFragmentManager, this) {
            when (it) {
                is PaymentBySavedCardFragment.Result.PaymentSuccess -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaymentSuccess(
                        paymentId = it.paymentId,
                        cardId = it.cardId,
                        rebillId = it.rebillId
                    )
                )

                is PaymentBySavedCardFragment.Result.PaymentError -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaymentError(
                        it.throwable
                    )
                )

                PaymentBySavedCardFragment.Result.PaymentCancel -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaymentCancel
                )

                is PaymentBySavedCardFragment.Result.NeedChangeCard -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.ChangeCard(
                        savedCardsOptions = it.savedCardsOptions,
                        paymentOptions = it.paymentOptions
                    )
                )

                is PaymentBySavedCardFragment.Result.NeedThreeDs -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.NeedThreeDs(
                        paymentOptions = it.paymentOptions,
                        data = it.data,
                        paymentSource = it.paymentSource
                    )
                )
            }
        }
        PaymentByNewCardFragment.registerResultListener(fragment.childFragmentManager, this) {
            when (it) {
                is PaymentByNewCardFragment.Result.PaymentSuccess -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaymentSuccess(
                        paymentId = it.paymentId,
                        cardId = it.cardId,
                        rebillId = it.rebillId
                    )
                )

                is PaymentByNewCardFragment.Result.PaymentError -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaymentError(
                        it.throwable
                    )
                )

                PaymentByNewCardFragment.Result.PaymentCancel -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaymentCancel
                )

                is PaymentByNewCardFragment.Result.NeedThreeDs -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.NeedThreeDs(
                        paymentOptions = it.paymentOptions,
                        data = it.data,
                        paymentSource = it.paymentSource
                    )
                )
            }
        }
        PaymentByCardFragment.registerResultListener(fragment.childFragmentManager, this) {
            when (it) {
                is PaymentByCardFragment.Result.PaymentSuccess -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaymentSuccess(
                        paymentId = it.paymentId,
                        cardId = it.cardId,
                        rebillId = it.rebillId
                    )
                )

                is PaymentByCardFragment.Result.PaymentError -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaymentError(
                        it.throwable
                    )
                )

                PaymentByCardFragment.Result.PaymentCancel -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaymentCancel
                )

                is PaymentByCardFragment.Result.NeedChangeCard -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.ChangeCard(
                        savedCardsOptions = it.savedCardsOptions,
                        paymentOptions = it.paymentOptions,
                        card = it.card
                    )
                )

                is PaymentByCardFragment.Result.NeedThreeDs -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.NeedThreeDs(
                        paymentOptions = it.paymentOptions,
                        data = it.data,
                        paymentSource = it.paymentSource
                    )
                )
            }
        }

        CardListFlowFragment.registerResultListener(fragment.childFragmentManager, this) {
            when (it) {
                CardListFlowFragment.Result.Cancel -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.ChangeCardCancel
                )

                is CardListFlowFragment.Result.CardChosen -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.ChangeCardSuccess(it.card)
                )

                is CardListFlowFragment.Result.Failed -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.ChangeCardFailed(it.error)
                )

                is CardListFlowFragment.Result.PaidByNewCard -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.PaidByNewCard(
                        paymentId = it.paymentId,
                        cardId = it.cardId,
                        rebillId = it.rebillId,
                    )
                )

                null -> Unit
            }
        }

        ThreeDsFragment.registerResultListener(fragment.childFragmentManager, this) {
            when (it) {
                ThreeDsResult.Cancel -> viewModel.handleEvent(PaymentByCardFlowViewModel.Event.ThreeDsCancel)
                is ThreeDsResult.Error -> viewModel.handleEvent(
                    PaymentByCardFlowViewModel.Event.ThreeDsError(
                        error = it.error,
                        paymentId = it.paymentId
                    )
                )

                is ThreeDsResult.Success -> {
                    val paymentResult = it.result as PaymentResult
                    viewModel.handleEvent(
                        PaymentByCardFlowViewModel.Event.ThreeDsSuccess(
                            paymentId = paymentResult.paymentId,
                            cardId = paymentResult.cardId,
                            rebillId = paymentResult.rebillId
                        )
                    )
                }

                null -> Unit
            }
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        viewBinding = null
        notificationBottomSheetBinding = null
    }

    private suspend fun openPaymentByCard() {
        val args = PaymentByCardFlowFragmentArgs.fromBundle(requireArguments())
        navHostFragment?.navController?.navigate(
            getPayByCardDirection(args.paymentOptions, args.card, args.showArrow)
        )
    }

    private fun returnResult(result: Result) {
        setFragmentResult(
            RESULT_KEY, bundleOf(
                RESULT_DATA_KEY to result
            )
        )
    }

    companion object {
        fun newInstance(
            paymentOptions: PaymentOptions,
            card: Card? = null,
            showArrow: Boolean = false,
            isNewCardViewToggleEnabled: Boolean,
        ): Fragment {
            return PaymentByCardFlowFragment().apply {
                arguments =
                    PaymentByCardFlowFragmentArgs(
                        paymentOptions = paymentOptions,
                        card = card,
                        showArrow = showArrow,
                        isNewCardViewToggleEnabled = isNewCardViewToggleEnabled
                    ).toBundle()
            }
        }

        private const val RESULT_KEY = "PaymentByCardFlowFragment.RESULT_KEY"
        private const val RESULT_DATA_KEY = "PaymentByCardFlowFragment.RESULT_DATA_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(RESULT_KEY, lifecycleOwner) { _, bundle ->
                listener.invoke(bundle.getParcelable(RESULT_DATA_KEY)!!)
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data class PaymentSuccess(
            var paymentId: Long,
            var cardId: String? = null,
            var rebillId: String? = null
        ) : Result

        @Parcelize
        data class PaymentError(
            val error: Throwable,
            val paymentId: Long?
        ) : Result

        @Parcelize
        data object PaymentCancel : Result
    }

    @Parcelize
    data class Args(
        val paymentOptions: PaymentOptions,
        val card: Card? = null,
        val showArrow: Boolean = false
    ) : Parcelable
}
