package ru.tinkoff.acquiring.sdk.redesign.payment.ui

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory

/**
 * @author s.y.biryukov
 */
class PaymentByCardFlowViewModel(
    savedStateHandle: SavedStateHandle
) : ViewModel() {

    private val _stateFlow = MutableStateFlow(State())
    val state = _stateFlow.asStateFlow()
    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()

    private val args = PaymentByCardFlowFragmentArgs.fromSavedStateHandle(savedStateHandle)

    fun handleEvent(event: Event) {
        when (event) {
            is Event.PaymentSuccess -> onPaymentSuccess(
                paymentId = event.paymentId,
                cardId = event.cardId,
                rebillId = event.rebillId
            )
            is Event.PaymentError -> onPaymentError(event.throwable)
            Event.PaymentCancel -> onPaymentCancel()
            is Event.ChangeCard -> onChangeCard(event)
            Event.ChangeCardCancel -> onChangeCardCancel()
            is Event.ChangeCardSuccess -> onChangeCardSuccess(event.card)
            is Event.ChangeCardFailed -> onCardChangeFailed(event.error)
            is Event.PaidByNewCard -> onPaidByNewCard(
                paymentId = event.paymentId,
                cardId = event.cardId,
                rebillId = event.rebillId,
            )

            is Event.NeedThreeDs -> onNeedThreeDs(
                event.paymentOptions,
                event.data,
                event.paymentSource
            )

            Event.ThreeDsCancel -> onThreeDsCancel()
            is Event.ThreeDsError -> onThreeDsError(event)
            is Event.ThreeDsSuccess -> onThreeDsSuccess(event)
        }
    }

    private fun onThreeDsSuccess(event: Event.ThreeDsSuccess) {
        if (args.paymentOptions.features.showPaymentNotifications) {
            val notification = NotificationFactory.getPaymentSuccessCommonNotification(
                amount = args.paymentOptions.order.amount
            ) {
                returnSuccessResult(
                    paymentId = event.paymentId,
                    cardId = event.cardId,
                    rebillId = event.rebillId
                )
            }
            _stateFlow.update {
                it.copy(
                    notification = notification
                )
            }
            viewModelScope.launch {
                _commandFlow.emit(Command.ReturnToEmptyScreen)
            }
        } else {
            returnSuccessResult(
                paymentId = event.paymentId,
                cardId = event.cardId,
                rebillId = event.rebillId
            )
        }
    }

    private fun onThreeDsError(event: Event.ThreeDsError) {
        val onClick = {
            _stateFlow.update {
                it.copy(notification = null)
            }
            returnErrorResult(event.error)
        }
        if (args.paymentOptions.features.showPaymentNotifications) {
            val notification = NotificationFactory.getPaymentErrorNotification(
                exception = event.error,
                onClickFactory = {
                    onClick
                },
                options = NotificationFactory.getNotificationOptions(args.paymentOptions.sdkContext)
            )
            _stateFlow.update {
                it.copy(
                    notification = notification
                )
            }
        } else {
            onClick()
        }
    }

    private fun onThreeDsCancel() {
        returnToPrevScreen()
    }

    private fun onNeedThreeDs(
        paymentOptions: PaymentOptions,
        data: ThreeDsData,
        paymentSource: PaymentSource
    ) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.CheckThreeDs(
                    paymentOptions = paymentOptions,
                    data = data,
                    paymentSource = paymentSource
                )
            )
        }
    }

    private fun onCardChangeFailed(error: Throwable) {
        returnErrorResult(error)
    }

    private fun onPaidByNewCard(paymentId: Long, cardId: String?, rebillId: String?) {
        returnSuccessResult(paymentId, cardId, rebillId)
    }

    private fun onChangeCardSuccess(card: Card) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.PayWithNewCard(
                    card = card,
                    paymentOptions = args.paymentOptions
                )
            )
        }
    }

    private fun onChangeCardCancel() {
        returnToPrevScreen()
    }

    private fun returnToPrevScreen() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnToPrevScreen)
        }
    }

    private fun onChangeCard(event: Event.ChangeCard) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ChangeCard(
                    savedCardsOptions = event.savedCardsOptions,
                    paymentOptions = event.paymentOptions,
                    card = event.card
                )
            )
        }
    }

    private fun onPaymentCancel() {
        viewModelScope.launch(Dispatchers.Main) {
            _commandFlow.emit(Command.ReturnPaymentCancelResult)
        }
    }

    private fun onPaymentError(throwable: Throwable) {
        returnErrorResult(throwable)
    }

    private fun returnErrorResult(error: Throwable) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnPaymentErrorResult(
                    error = error,
                    paymentId = args.paymentOptions.paymentId
                )
            )
        }
    }

    private fun onPaymentSuccess(paymentId: Long, cardId: String?, rebillId: String?) {
        returnSuccessResult(paymentId, cardId, rebillId)
    }

    private fun returnSuccessResult(paymentId: Long, cardId: String?, rebillId: String?) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnPaymentSuccessResult(
                    paymentId = paymentId,
                    cardId = cardId,
                    rebillId = rebillId
                )
            )
        }
    }

    data class State(
        val notification: Notification? = null
    )


    sealed interface Event {

        data class PaymentSuccess(
            val paymentId: Long,
            val cardId: String? = null,
            val rebillId: String? = null
        ) : Event

        data class PaymentError(
            val throwable: Throwable
        ) : Event

        data class ChangeCard(
            val savedCardsOptions: SavedCardsOptions,
            val paymentOptions: PaymentOptions,
            val card: Card? = null
        ) : Event

        data class ChangeCardSuccess(val card: Card) : Event

        data class PaidByNewCard(
            val paymentId: Long,
            val cardId: String?,
            val rebillId: String?
        ) : Event

        data class ChangeCardFailed(
            val error: Throwable
        ) : Event

        data class NeedThreeDs(
            val paymentOptions: PaymentOptions,
            val data: ThreeDsData,
            val paymentSource: PaymentSource
        ) : Event

        data class ThreeDsError(
            val error: Throwable,
            val paymentId: Long?
        ) : Event

        data class ThreeDsSuccess(
            var paymentId: Long,
            var cardId: String? = null,
            var rebillId: String? = null
        ) : Event

        data object PaymentCancel : Event
        data object ChangeCardCancel : Event
        data object ThreeDsCancel : Event
    }

    sealed interface Command {
        data object ReturnToEmptyScreen : Command
        data object ReturnPaymentCancelResult : Command
        data object ReturnToPrevScreen : Command
        data class PayWithNewCard(
            val paymentOptions: PaymentOptions,
            val card: Card
        ) : Command

        data class ReturnPaymentSuccessResult(
            var paymentId: Long,
            var cardId: String? = null,
            var rebillId: String? = null
        ) : Command

        data class ReturnPaymentErrorResult(
            val error: Throwable,
            val paymentId: Long?
        ) : Command

        data class ChangeCard(
            val savedCardsOptions: SavedCardsOptions,
            val paymentOptions: PaymentOptions,
            val card: Card?
        ) : Command

        data class CheckThreeDs(
            val paymentOptions: PaymentOptions,
            val data: ThreeDsData,
            val paymentSource: PaymentSource
        ) : Command
    }

}
