package ru.tinkoff.acquiring.sdk.redesign.payment.ui

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.addCallback
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.widget.SwitchCompat
import androidx.cardview.widget.CardView
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.core.view.postDelayed
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentContainerView
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.commit
import androidx.fragment.app.setFragmentResult
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.navArgs
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.mapNotNull
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentPaymentByCardBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.BaseAcquiringOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.redesign.common.carddatainput.CardDataInputFragment
import ru.tinkoff.acquiring.sdk.redesign.common.emailinput.EmailInputFragment
import ru.tinkoff.acquiring.sdk.ui.component.bindKtx
import ru.tinkoff.acquiring.sdk.ui.customview.LoaderButton
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.keyboard.SecureKeyboardController
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegate
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegateImpl
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.utils.lazyUnsafe
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

/**
 * Created by v.budnitskiy
 */
internal class PaymentByCardFragment : Fragment(),
    CardDataInputFragment.OnCardDataChanged,
    EmailInputFragment.OnEmailDataChanged,
    AppBaseChallengeDelegate by AppBaseChallengeDelegateImpl(), IsolatedKoinComponent {

    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? =
        null
    private var viewBinding: AcqFragmentPaymentByCardBinding? = null

    private val args: PaymentByCardFragmentArgs by navArgs()

    private val paymentOptions by lazy { args.paymentOptions }

    private val isCardSaved: Boolean by lazy {
        viewModel.isCardSavedFocusChanger()
    }

    private var cardDataInputFragment:CardDataInputFragment? = null
    private var chosenCardComponent: ChosenCardComponent? = null
    private val emailInput: EmailInputFragment by lazyUnsafe {
        EmailInputFragment.getInstance(paymentOptions.customer.email)
    }
    private val emailInputContainer: FragmentContainerView? get() = view?.findViewById(R.id.fragment_email_input)
    private val sendReceiptSwitch: SwitchCompat? get() = view?.findViewById(R.id.acq_send_receipt_switch)
    private val payButton: LoaderButton? get() = view?.findViewById(R.id.acq_pay_btn)

    private val viewModel by viewModel<PaymentByCardViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return AcqFragmentPaymentByCardBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        initToolbar()
        initViews()
        initAppBaseChallengeDelegate(
            requireActivity(), AcquiringSdk(
                paymentOptions.terminalKey,
                paymentOptions.publicKey
            ),
            viewLifecycleOwner.lifecycleScope
        )

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.handleEvent(PaymentByCardViewModel.Event.BackPressed)
        }

        val chosenCardComponent = ChosenCardComponent(
            view.findViewById<CardView?>(R.id.acq_chosen_card),
            isCardSaved,
            onChangeCard = { onChangeCard() },
            onCvcCompleted = { cvc, isValid ->
                viewModel.handleEvent(
                    PaymentByCardViewModel.Event.CvcEnter(
                        cvc = cvc,
                        isValid = isValid
                    )
                )
            },
            useSecureKeyboard = paymentOptions.features.useSecureKeyboard,
        )
        chosenCardComponent.clearCvc()
        this.chosenCardComponent = chosenCardComponent
        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                uiState()
            }
        }

        updateCardDataInputFragment(paymentOptions)

        chosenCardComponent.bindKtx(lifecycleScope, viewModel.stateFlow.mapNotNull { it.chosenCard })

        this.notificationBottomSheetBinding = NotificationUtils.bindNotificationBottomSheet(
            activity = requireActivity()
        )

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.CREATED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is PaymentByCardViewModel.Command.ChangeCard -> {
                            returnResult(
                                Result.NeedChangeCard(
                                    savedCardsOptions = it.savedCardsOptions,
                                    paymentOptions = it.paymentOptions,
                                    card = it.card
                                )
                            )
                        }

                        is PaymentByCardViewModel.Command.RequestThreeDs -> {
                            returnResult(
                                Result.NeedThreeDs(
                                    paymentOptions = paymentOptions,
                                    data = it.data,
                                    paymentSource = it.paymentSource
                                )
                            )
                        }

                        is PaymentByCardViewModel.Command.RequestThreeAppBaseChallenge -> {
                            launchChallenge(
                                transaction = it.transaction,
                                threeDsData = it.threeDsData,
                                onResult = viewModel::processChallengeResult
                            )
                        }

                        PaymentByCardViewModel.Command.ReturnPaymentCancel -> {
                            finishWithCancel()
                        }

                        is PaymentByCardViewModel.Command.ReturnPaymentError -> {
                            finishWithError(it.throwable)
                        }

                        is PaymentByCardViewModel.Command.ReturnPaymentSuccess -> {
                            finishWithSuccess(
                                paymentId = it.paymentId,
                                cardId = it.cardId,
                                rebillId = it.rebillId
                            )
                        }

                        PaymentByCardViewModel.Command.RequestEmailFieldFocus -> {
                            if (emailInput.isAdded && emailInput.isVisible) {
                                emailInput.viewModel.requestEmailInputFocus()
                            }
                        }

                        PaymentByCardViewModel.Command.RequestCardFieldFocus -> {
                            setFocusToCardNumber()
                        }

                        PaymentByCardViewModel.Command.ClearFocus -> {
                            if (emailInput.isAdded && emailInput.isVisible) {
                                emailInput.viewModel.clearEmailInputFocus()
                                emailInput.viewModel.hideEmailInputKeyboard()
                            }
                            clearFocusFromCardNumber()
                        }
                    }
                }
            }
        }
    }

    private fun clearFocusFromCardNumber() {
        val cardDataInput = cardDataInputFragment ?: return
        if (cardDataInput.isAdded && cardDataInput.isVisible) {
            cardDataInput.clearFocus()
        }
    }

    private fun setFocusToCardNumber() {
        requireView().postDelayed(500) { // TODO Убрать после переработки CardDataInputFragment
            val cardDataInput = cardDataInputFragment
            if (cardDataInput != null && cardDataInput.isAdded && cardDataInput.isVisible) {
                cardDataInput.requestFocusOnCardNumber()
            }
        }
    }

    private fun updateCardDataInputFragment(options: BaseAcquiringOptions) {
        if (cardDataInputFragment == null) {
            var fragment =
                childFragmentManager.findFragmentById(R.id.fragment_card_data_input) as? CardDataInputFragment
            if (fragment == null) {
                fragment = CardDataInputFragment.newInstance(options)
                childFragmentManager.commit {
                    add(R.id.fragment_card_data_input, fragment)
                    addToBackStack(null)
                }
            }
            cardDataInputFragment = fragment
        }
    }

    override fun onResume() {
        super.onResume()

        viewBinding?.let {
            SecureKeyboardController.getInstance().apply {
                setContentContainer(it.acqPayByCardContentWrapper)
                registerInsets(it.root)
            }
        }
        viewModel.handleEvent(PaymentByCardViewModel.Event.Resume)
    }

    override fun onPause() {
        super.onPause()
        SecureKeyboardController.getInstance().clear()
    }

    override fun onDestroy() {
        super.onDestroy()
        (requireActivity() as? AppCompatActivity)?.setSupportActionBar(null)
    }

    override fun onDestroyView() {
        viewBinding = null
        cardDataInputFragment = null
        chosenCardComponent = null
        super.onDestroyView()
    }
    //endregion

    //region Data change callbacks
    override fun onCardDataChanged(
        data: CardDataInputFragment.OnCardDataChanged.Data,
        isValid: Boolean
    ) {
        viewModel.setCardData(
            cardNumber = data.pan,
            cvc = data.securityCode,
            dateExpired = data.expiryDate,
            isValidCardData = isValid
        )
    }

    override fun onEmailDataChanged(email: String, isValid: Boolean) {
        viewModel.setEmail(email = email, isValidEmail = isValid)
    }
    //endregion

    //region Navigation
    private fun onChangeCard() {
        viewModel.handleEvent(PaymentByCardViewModel.Event.ChangeCard)
    }

    private fun finishWithCancel() {
        returnResult(Result.PaymentCancel)
    }

    private fun finishWithError(throwable: Throwable) {
        returnResult(Result.PaymentError(throwable))
    }

    private fun finishWithSuccess(paymentId: Long, cardId: String?, rebillId: String?) {
        returnResult(
            Result.PaymentSuccess(
                paymentId = paymentId,
                cardId = cardId,
                rebillId = rebillId,
            )
        )
    }
    //endregion

    //region init views
    private fun initToolbar() {
        val toolbar = viewBinding?.acqToolbar
        val activity = (requireActivity() as? AppCompatActivity)
        activity?.setSupportActionBar(toolbar)
        activity?.supportActionBar?.setDisplayHomeAsUpEnabled(true)
        activity?.supportActionBar?.setDisplayShowHomeEnabled(true)
        activity?.supportActionBar?.setTitle(R.string.acq_cardpay_title)
        if (args.showArrow) {
            toolbar?.setNavigationIcon(R.drawable.acq_arrow_back)
        }
        toolbar?.setNavigationOnClickListener {
            viewModel.handleEvent(PaymentByCardViewModel.Event.BackPressed)
        }
    }

    private fun initViews() {

        emailInputContainer?.let {
            childFragmentManager.commit {
                replace(it.id, emailInput)
            }
        }

        sendReceiptSwitch?.setOnCheckedChangeListener { button, isChecked ->
            viewModel.handleEvent(
                PaymentByCardViewModel.Event.SendReceiptChange(
                    isChecked,
                    button.isPressed
                )
            )
        }

        payButton?.setOnClickListener {
            viewModel.handleEvent(PaymentByCardViewModel.Event.PayButtonClick)
        }
    }
    //endregion

    //region subscribe States
    private suspend fun uiState() {
        viewModel.stateFlow.collectLatest {
            viewBinding?.acqChosenCard?.root?.isVisible = it.chosenCard != null
            viewBinding?.fragmentCardDataInput?.isVisible = it.chosenCard == null
            emailInputContainer?.isVisible = it.sendReceipt
            sendReceiptSwitch?.isChecked = it.sendReceipt
            payButton?.text = getString(R.string.acq_cardpay_pay, it.amount)
            payButton?.isEnabled = it.buttonEnabled

            handleNotification(it.notification)

            handleLoadingInProcess(it.progress)
        }
    }

    private fun handleNotification(notification: Notification?) {
        if (notification != null) {
            notificationBottomSheetBinding?.show(
                notification.toStatusViewData(requireContext())
            )
        } else {
            notificationBottomSheetBinding?.hide()
        }
    }
    //endregion

    private fun handleLoadingInProcess(inProcess: Boolean) {
        payButton?.apply {
            isLoading = inProcess
            isClickable = !inProcess
        }
        emailInput.viewModel.enableEmail(!inProcess)
        chosenCardComponent?.enableCvc(!inProcess)
    }

    private fun returnResult(result: Result) {
        setFragmentResult(
            RESULT_KEY, bundleOf(
                RESULT_DATA_KEY to result
            )
        )
    }

    companion object {
        private const val RESULT_KEY = "PaymentByCardFragment.RESULT_KEY"
        private const val RESULT_DATA_KEY = "PaymentByCardFragment.RESULT_DATA_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(RESULT_KEY, lifecycleOwner) { _, bundle ->
                listener.invoke(bundle.getParcelable(RESULT_DATA_KEY)!!)
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data class PaymentSuccess(
            var paymentId: Long,
            var cardId: String? = null,
            var rebillId: String? = null
        ) : Result

        @Parcelize
        data class PaymentError(
            val throwable: Throwable
        ) : Result

        @Parcelize
        data class NeedThreeDs(
            val paymentOptions: PaymentOptions,
            val data: ThreeDsData,
            val paymentSource: PaymentSource,
        ) : Result

        @Parcelize
        data class NeedChangeCard(
            val savedCardsOptions: SavedCardsOptions,
            val paymentOptions: PaymentOptions,
            val card: Card?
        ) : Result

        @Parcelize
        data object PaymentCancel : Result
    }
}

