package ru.tinkoff.acquiring.sdk.redesign.payment.ui

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.models.paysources.AttachedCard
import ru.tinkoff.acquiring.sdk.models.paysources.CardData
import ru.tinkoff.acquiring.sdk.models.paysources.CardSource
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardProcess
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardState
import ru.tinkoff.acquiring.sdk.redesign.common.emailinput.models.EmailValidator
import ru.tinkoff.acquiring.sdk.redesign.payment.model.CardChosenModel
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBasedTransaction
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeResult
import ru.tinkoff.acquiring.sdk.utils.BankCaptionProvider
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory
import ru.tinkoff.acquiring.sdk.utils.lazyUnsafe

internal class PaymentByCardViewModel(
    savedStateHandle: SavedStateHandle,
    private val paymentByCardProcess: PaymentByCardProcess,
    private val bankCaptionProvider: BankCaptionProvider,
) : ViewModel() {

    private val args: PaymentByCardFragmentArgs =
        PaymentByCardFragmentArgs.fromSavedStateHandle(savedStateHandle)

    private val savedCardsOptions: SavedCardsOptions by lazyUnsafe {
        SavedCardsOptions().apply {
            setTerminalParams(
                paymentOptions.terminalKey,
                paymentOptions.publicKey
            )
            customer = paymentOptions.customer
            features = paymentOptions.features
            mode = SavedCardsOptions.Mode.PAYMENT
            withArrowBack = true
        }
    }

    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()


    private val chosenCard by lazyUnsafe {
        args.card?.let {
            CardChosenModel(it, bankCaptionProvider(it.pan))
        }
    }

    private val _isCardSaved = MutableStateFlow<Boolean?>(chosenCard != null)
    var isCardSaved: StateFlow<Boolean?> = _isCardSaved.asStateFlow()

    fun isCardSavedFocusChanger(): Boolean {
        return isCardSaved.value == true
    }

    private val paymentOptions = args.paymentOptions

    private val _stateFlow: MutableStateFlow<State> =
        MutableStateFlow(
            State(
                cardId = chosenCard?.id,
                isValidEmail = paymentOptions.customer.email?.let { EmailValidator.validate(it) } ?: true,
                sendReceipt = paymentOptions.customer.email.isNullOrBlank().not(),
                email = paymentOptions.customer.email,
                paymentOptions = paymentOptions,
                chosenCard = chosenCard
            )
        )
    val stateFlow: StateFlow<State> = _stateFlow.asStateFlow()

    private var isFirstTime: Boolean = true

    // ручной ввод карты
    fun setCardData(
        cardNumber: String? = null,
        cvc: String? = null,
        dateExpired: String? = null,
        isValidCardData: Boolean = false,
    ) {
        if (_stateFlow.value.chosenCard != null) return

        _stateFlow.update {
            it.copy(
                cardNumber = cardNumber,
                cvc = cvc,
                dateExpired = dateExpired,
                isValidCardData = isValidCardData,
                cardId = null,
            )
        }
    }

    // ввод кода сохраненной карты
    private fun setCvc(cvc: String, isValid: Boolean) =
        _stateFlow.update { it.copy(cvc = cvc, isValidCardData = isValid) }

    private fun sendReceiptChange(isSelect: Boolean, byUser:Boolean) {
        _stateFlow.update {
            it.copy(
                sendReceipt = isSelect,
                isValidEmail = EmailValidator.validate(it.email)
            )
        }

        if (byUser) {
            if (isSelect) {
                viewModelScope.launch {
                    _commandFlow.emit(Command.RequestEmailFieldFocus)
                }
            } else {
                viewModelScope.launch {
                    _commandFlow.emit(Command.ClearFocus)
                }
            }
        }
    }

    fun setEmail(email: String?, isValidEmail: Boolean) = _stateFlow.update {
        it.copy(email = email, isValidEmail = isValidEmail)
    }

    fun processChallengeResult(result: AppBaseChallengeResult) = viewModelScope.launch {
        when (result) {
            AppBaseChallengeResult.Cancelled -> {
                _commandFlow.emit(Command.ReturnPaymentCancel)
            }

            is AppBaseChallengeResult.ProtocolError -> {
                onChallengeError(result.error)
            }

            is AppBaseChallengeResult.RuntimeError -> {
                onChallengeError(result.error)
            }

            is AppBaseChallengeResult.Error -> {
                onChallengeError(result.error)
            }

            is AppBaseChallengeResult.Success -> {
                returnSuccessResultOrShowNotification(
                    paymentId = result.paymentResult.paymentId,
                    cardId = result.paymentResult.cardId,
                    rebillId = result.paymentResult.rebillId
                )
            }

            is AppBaseChallengeResult.TimeOut -> {
                val onClick: () -> Unit = {
                    viewModelScope.launch {
                        _stateFlow.update {
                            it.copy(
                                notification = null
                            )
                        }
                        _commandFlow.emit(Command.ReturnPaymentError(result.error))
                    }
                }
                val notification = NotificationFactory.getPaymentTimeoutErrorNotification(
                    exception = result.error,
                    onClickFactory = { onClick }
                )
                _stateFlow.update {
                    it.copy(
                        notification = notification
                    )
                }
            }

            AppBaseChallengeResult.Loading -> {
                val notification = Notification(
                    type = StatusViewData.Type.PROGRESS,
                    title = R.string.acq_commonsheet_processing_title,
                    description = R.string.acq_commonsheet_processing_description,
                )
                _stateFlow.update {
                    it.copy(
                        notification = notification
                    )
                }
            }
        }
    }

    private suspend fun onChallengeError(error: Throwable) {

        val onClick: () -> Unit = {
            viewModelScope.launch {
                _commandFlow.emit(Command.ReturnPaymentError(error))
            }
        }

        _stateFlow.update {
            it.copy(
                notification = NotificationFactory.getPaymentErrorNotification(
                    exception = error,
                    onClickFactory = { onClick },
                    options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
                )
            )
        }
    }

    private fun pay() {
        val stateValue = _stateFlow.value
        val emailForPayment = if (stateValue.sendReceipt) stateValue.email else null

        viewModelScope.launch {
            paymentByCardProcess.state.collectLatest {
                when (it) {
                    is PaymentByCardState.Started -> setProgress(true)
                    is PaymentByCardState.Created -> Unit
                    is PaymentByCardState.Error -> returnErrorOrShowNotification(it.throwable)
                    is PaymentByCardState.Success -> {
                        returnSuccessResultOrShowNotification(
                            paymentId = it.paymentId,
                            cardId = it.cardId,
                            rebillId = it.rebillId
                        )
                    }

                    is PaymentByCardState.ThreeDsUiNeeded -> {
                        viewModelScope.launch {
                            _commandFlow.emit(
                                Command.RequestThreeDs(
                                    paymentOptions = paymentOptions,
                                    data = it.threeDsState.data,
                                    paymentSource = _stateFlow.value.cardSource
                                )
                            )
                        }
                        goTo3ds()
                    }

                    is PaymentByCardState.ThreeDsInProcess -> setProgress(false)

                    is PaymentByCardState.ThreeDsAppBase -> {
                        viewModelScope.launch {
                            setProgress(true)
                            _commandFlow.emit(
                                Command.RequestThreeAppBaseChallenge(
                                    transaction = it.transaction,
                                    threeDsData = it.threeDsData,
                                )
                            )
                        }
                    }

                    else -> Unit
                }
            }
        }

        paymentByCardProcess.start(
            cardData = stateValue.cardSource,
            paymentOptions = stateValue.paymentOptions,
            email = emailForPayment
        )
    }

    private fun returnErrorOrShowNotification(error: Throwable) {
        val onClick = {
            _stateFlow.update {
                it.copy(notification = null)
            }
            returnErrorResult(error)
        }
        if (paymentOptions.features.showPaymentNotifications) {
            val notification = NotificationFactory.getPaymentErrorNotification(
                exception = error,
                onClickFactory = { onClick },
                options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
            )
            _stateFlow.update {
                it.copy(
                    notification = notification
                )
            }
        } else {
            onClick()
        }
    }

    private fun returnSuccessResultOrShowNotification(paymentId: Long, cardId: String?, rebillId: String?) {
        val onCLick = {
            returnSuccessResult(
                paymentId = paymentId,
                cardId = cardId,
                rebillId = rebillId
            )
        }
        if (paymentOptions.features.showPaymentNotifications) {
            val notification =
                NotificationFactory.getPaymentSuccessCommonNotification(
                    amount = paymentOptions.order.amount,
                    onClick = onCLick
                )
            _stateFlow.update {
                it.copy(
                    notification = notification
                )
            }
        } else {
            onCLick()
        }
    }

    private fun returnSuccessResult(paymentId: Long, cardId: String?, rebillId: String?) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnPaymentSuccess(
                    paymentId = paymentId,
                    cardId = cardId,
                    rebillId = rebillId
                )
            )
        }
    }

    private fun returnErrorResult(throwable: Throwable) {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnPaymentError(throwable))
        }
    }

    private fun setProgress(progress: Boolean) {
        _stateFlow.update {
            it.copy(progress = progress)
        }
    }

    private fun goTo3ds() {
        paymentByCardProcess.goTo3ds()
    }

    private fun cancelPayment() {
        paymentByCardProcess.stop()
    }

    data class State(
        val progress: Boolean = false,
        val cardId: String? = null,
        private val cardNumber: String? = null,
        val cvc: String? = null,
        private val dateExpired: String? = null,
        private val isValidCardData: Boolean = false,
        val isValidEmail: Boolean = false,
        val chosenCard: CardChosenModel? = null,
        val sendReceipt: Boolean = false,
        val email: String? = null,
        val paymentOptions: PaymentOptions,
        val notification: Notification? = null,
        val requestCardFieldFocusCommand: Boolean = false
    ) {

        val buttonEnabled: Boolean = if (sendReceipt) {
            isValidCardData && isValidEmail
        } else {
            isValidCardData
        }

        val amount = paymentOptions.order.amount.toHumanReadableString()

        val cardSource: CardSource
            get() {
                return if (cardId != null) {
                    AttachedCard(cardId, cvc)
                } else {
                    CardData(cardNumber!!, dateExpired!!, cvc!!)
                }
            }
    }

    fun handleEvent(event: Event) {
        when (event) {
            Event.ChangeCard -> requestCardChange()
            Event.BackPressed -> onBackPressed()
            is Event.CvcEnter -> setCvc(event.cvc, event.isValid)
            Event.PayButtonClick -> onPayButtonClick()
            is Event.SendReceiptChange -> sendReceiptChange(event.isChecked, event.isChangedByUser)
            Event.Resume -> onViewResume()
        }
    }

    private fun onViewResume() {
        if (isFirstTime) {
            onViewResumeFirstTime()
        }
    }

    private fun onViewResumeFirstTime() {
        isFirstTime = false
        if (chosenCard == null) {
            viewModelScope.launch {
                _commandFlow.emit(Command.RequestCardFieldFocus)
            }
        }
    }

    private fun onPayButtonClick() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ClearFocus)

            pay()
        }
    }

    private fun onBackPressed() {
        cancelPayment()
        returnPaymentCancel()
    }

    private fun returnPaymentCancel() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnPaymentCancel)
        }
    }

    private fun requestCardChange() {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ChangeCard(
                    savedCardsOptions = savedCardsOptions,
                    card = args.card,
                    paymentOptions = paymentOptions
                )
            )
        }
    }

    sealed interface Event {
        data object Resume : Event
        class CvcEnter(
            val cvc: String,
            val isValid: Boolean
        ) : Event

        data object ChangeCard : Event
        data object BackPressed : Event
        data object PayButtonClick : Event

        class SendReceiptChange(
            val isChecked: Boolean,
            val isChangedByUser: Boolean,
        ) : Event
    }

    sealed interface Command {
        class ReturnPaymentError(
            val throwable: Throwable
        ) : Command

        class RequestThreeDs(
            val paymentOptions: PaymentOptions,
            val data: ThreeDsData,
            val paymentSource: PaymentSource
        ) : Command

        class RequestThreeAppBaseChallenge(
            val transaction: ThreeDsAppBasedTransaction,
            val threeDsData: ThreeDsData,
        ) : Command

        class ReturnPaymentSuccess(
            val paymentId: Long,
            val cardId: String?,
            val rebillId: String?
        ) : Command

        data class ChangeCard(
            val savedCardsOptions: SavedCardsOptions,
            val paymentOptions: PaymentOptions,
            val card: Card?,
        ) : Command

        data object ReturnPaymentCancel : Command
        data object RequestEmailFieldFocus : Command
        data object RequestCardFieldFocus : Command
        data object ClearFocus : Command
    }
}
