package ru.tinkoff.acquiring.sdk.redesign.payment.ui.v2

import android.content.Context
import android.text.method.PasswordTransformationMethod
import android.util.AttributeSet
import android.view.LayoutInflater
import android.widget.FrameLayout
import ru.tinkoff.acquiring.sdk.cardscanners.delegate.CardScannerDelegate
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentCardDataInputBinding
import ru.tinkoff.acquiring.sdk.redesign.common.carddatainput.CardDataInputFragment.Companion.createCvcMask
import ru.tinkoff.acquiring.sdk.redesign.common.carddatainput.CardDataInputFragment.Companion.createExpiryDateMask
import ru.tinkoff.acquiring.sdk.redesign.common.carddatainput.CardNumberFormatter
import ru.tinkoff.acquiring.sdk.smartfield.BaubleCardLogo
import ru.tinkoff.acquiring.sdk.smartfield.BaubleClearButton
import ru.tinkoff.acquiring.sdk.smartfield.BaubleClearOrScanButton
import ru.tinkoff.acquiring.sdk.utils.SimpleTextWatcher.Companion.afterTextChanged
import ru.tinkoff.decoro.watchers.MaskFormatWatcher

/**
 * @author s.y.biryukov
 */
class CardDataInputView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {
    private val viewBinding =
        AcqFragmentCardDataInputBinding.inflate(LayoutInflater.from(context), this, true)
    private val cardScannerDelegate: CardScannerDelegate = object: CardScannerDelegate {
        override val isEnabled: Boolean get() = isScanEnabled

        override fun start() {

        }
    }

    var isScanEnabled: Boolean = false

    var listener: Listener? = null

    var useSecuredKeyboard: Boolean = false
        set(value) {
            field = value
            viewBinding.cardNumberInput.useSecureKeyboard = value
            viewBinding.expiryDateInput.useSecureKeyboard = value
            viewBinding.cvcInput.useSecureKeyboard = value
        }

    init {
        viewBinding.cardNumberInput.apply {
            BaubleCardLogo().attach(this)
            val cardNumberFormatter = CardNumberFormatter()
            editText.addTextChangedListener(cardNumberFormatter)
            textInputLayout.addFocusChangeListener { notifyFocusChanged() }
            val clearOrScanButton = BaubleClearOrScanButton()
            clearOrScanButton.onScanButtonClick = {
                listener?.onScanButtonClick()
            }
            clearOrScanButton.attach(this, cardScannerDelegate)
            editText.afterTextChanged {
                listener?.onCardNumberChanged(it.toString())
            }
        }

        viewBinding.expiryDateInput.apply {
            MaskFormatWatcher(createExpiryDateMask()).installOn(editText)
            BaubleClearButton().attach(this)
            textInputLayout.addFocusChangeListener { notifyFocusChanged() }
            editText.afterTextChanged {
                listener?.onExpireDateChanged(it.toString())
            }
        }

        viewBinding.cvcInput.apply {
            transformationMethod = PasswordTransformationMethod()
            MaskFormatWatcher(createCvcMask()).installOn(editText)
            BaubleClearButton().attach(this)
            textInputLayout.addFocusChangeListener { notifyFocusChanged() }
            editText.afterTextChanged {
                listener?.onCvcChanged(it.toString())
            }
        }
    }

    fun showCardNumberError(isError: Boolean) {
        viewBinding.cardNumberInput.apply {
            errorHighlighted = isError
        }
    }

    fun expireDateError(isError: Boolean) {
        viewBinding.expiryDateInput.apply {
            errorHighlighted = isError
        }
    }

    fun showCvcError(isError: Boolean) {
        viewBinding.cvcInput.apply {
            errorHighlighted = isError
        }
    }

    fun focusNumberField() {
        viewBinding.cardNumberInput.requestViewFocus()
    }

    fun focusCvcField() {
        viewBinding.cvcInput.requestViewFocus()
    }

    fun focusExpireDateField() {
        viewBinding.expiryDateInput.requestViewFocus()
    }

    fun setNumber(cardNumber: String?) {
        viewBinding.cardNumberInput.text = cardNumber
    }

    fun setExpireDate(expireDate: String?) {
        viewBinding.expiryDateInput.text = expireDate
    }

    fun setCvc(cvc: String?) {
        viewBinding.cvcInput.text = cvc
    }

    override fun setEnabled(enabled: Boolean) {
        super.setEnabled(enabled)
        viewBinding.cardNumberInput.apply {
            this.editable = enabled
            this.isEnabled = enabled
        }
        viewBinding.cvcInput.apply {
            this.editable = enabled
            this.isEnabled = enabled
        }
        viewBinding.expiryDateInput.apply {
            this.editable = enabled
            this.isEnabled = enabled
        }
    }

    private fun notifyFocusChanged() {
        listener?.onFocusChanged(
            isCardNumberFocused = viewBinding.cardNumberInput.isViewFocused(),
            isExpireDateFocused = viewBinding.expiryDateInput.isViewFocused(),
            isCvcFocused = viewBinding.cvcInput.isViewFocused()
        )
    }

    interface Listener {
        fun onCardNumberChanged(cardNumber: String)
        fun onExpireDateChanged(expireDate: String)
        fun onCvcChanged(cvc: String)
        fun onScanButtonClick()
        fun onFocusChanged(isCardNumberFocused: Boolean, isExpireDateFocused: Boolean, isCvcFocused: Boolean)
    }
}
