package ru.tinkoff.acquiring.sdk.redesign.payment.ui.v2

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.addCallback
import androidx.appcompat.app.AppCompatActivity
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.setFragmentResult
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.cardscanners.delegate.CardScannerWrapper
import ru.tinkoff.acquiring.sdk.cardscanners.delegate.ScannedCardResult
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentPaymentByNewCardBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.keyboard.SecureKeyboardController
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegate
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegateImpl
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.utils.KeyboardVisionUtils
import ru.tinkoff.acquiring.sdk.utils.setDisabledFieldAlpha
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

/**
 * Created by v.budnitskiy
 */
internal class PaymentByNewCardFragment : Fragment(),
    AppBaseChallengeDelegate by AppBaseChallengeDelegateImpl(), IsolatedKoinComponent {

    private var cardScannerWrapper: CardScannerWrapper? = null
    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? =
        null
    private var viewBinding: AcqFragmentPaymentByNewCardBinding? = null

    private val viewModel by viewModel<PaymentByNewCardViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        this.cardScannerWrapper = CardScannerWrapper(requireActivity(), ::onCardScannerResult)
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return AcqFragmentPaymentByNewCardBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.handleEvent(PaymentByNewCardViewModel.Event.BackPressed)
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                collectState()
            }
        }

        this.notificationBottomSheetBinding = NotificationUtils.bindNotificationBottomSheet(
            activity = requireActivity()
        )

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.CREATED) {
                collectCommand()
            }
        }
    }

    override fun onResume() {
        super.onResume()

        viewBinding?.let {
            SecureKeyboardController.getInstance().apply {
                setContentContainer(it.acqPayByCardContentWrapper)
                registerInsets(it.root)
            }
        }
        viewModel.handleEvent(PaymentByNewCardViewModel.Event.Resume)
    }

    override fun onPause() {
        super.onPause()
        SecureKeyboardController.getInstance().clear()
    }

    override fun onDestroy() {
        super.onDestroy()
        (requireActivity() as? AppCompatActivity)?.setSupportActionBar(null)
    }

    override fun onDestroyView() {
        viewBinding = null
        super.onDestroyView()
    }

    private suspend fun collectCommand(): Nothing {
        viewModel.commandFlow.collect {
            when (it) {
                is PaymentByNewCardViewModel.Command.RequestThreeDs -> {
                    returnResult(
                        Result.NeedThreeDs(
                            paymentOptions = it.paymentOptions,
                            data = it.data,
                            paymentSource = it.paymentSource
                        )
                    )
                }

                is PaymentByNewCardViewModel.Command.RequestThreeAppBaseChallenge -> {
                    launchChallenge(
                        transaction = it.transaction,
                        threeDsData = it.threeDsData,
                        onResult = viewModel::processChallengeResult
                    )
                }

                PaymentByNewCardViewModel.Command.ReturnPaymentCancel -> {
                    finishWithCancel()
                }

                is PaymentByNewCardViewModel.Command.ReturnPaymentError -> {
                    finishWithError(it.throwable)
                }

                is PaymentByNewCardViewModel.Command.ReturnPaymentSuccess -> {
                    finishWithSuccess(
                        paymentId = it.paymentId,
                        cardId = it.cardId,
                        rebillId = it.rebillId
                    )
                }

                PaymentByNewCardViewModel.Command.RequestEmailFieldFocus -> {
                    viewBinding?.emailInput?.requestFocus()
                }

                PaymentByNewCardViewModel.Command.RequestCardNumberFieldFocus -> {
                    setFocusToCardNumber()
                }

                PaymentByNewCardViewModel.Command.ClearFocus -> {
                    KeyboardVisionUtils.hideKeyboard(requireView())
                    viewBinding?.emailInput?.clearFocus()
                    viewBinding?.newCardDataInput?.clearFocus()
                }

                PaymentByNewCardViewModel.Command.RequestCvcFieldFocus -> {
                    viewBinding?.newCardDataInput?.focusCvcField()
                }

                PaymentByNewCardViewModel.Command.RequestExpireDateFieldFocus -> {
                    viewBinding?.newCardDataInput?.focusExpireDateField()
                }

                is PaymentByNewCardViewModel.Command.FillCardData -> {
                    viewBinding?.newCardDataInput?.apply {
                        setNumber(it.cardNumber)
                        setExpireDate(it.expireDate)
                        setCvc(it.cvc)
                    }
                }

                is PaymentByNewCardViewModel.Command.FillEmailField -> {
                    viewBinding?.emailInput?.setData(it.email)
                }
            }
        }
    }

    private fun setFocusToCardNumber() {
        viewBinding?.newCardDataInput?.focusNumberField()
    }

    private fun finishWithCancel() {
        returnResult(Result.PaymentCancel)
    }

    private fun finishWithError(throwable: Throwable) {
        returnResult(Result.PaymentError(throwable))
    }

    private fun finishWithSuccess(paymentId: Long, cardId: String?, rebillId: String?) {
        returnResult(
            Result.PaymentSuccess(
                paymentId = paymentId,
                cardId = cardId,
                rebillId = rebillId,
            )
        )
    }

    private fun initToolbar(state: PaymentByNewCardViewModel.State) {
        val toolbar = viewBinding?.acqToolbar
        val activity = (requireActivity() as? AppCompatActivity)
        activity?.setSupportActionBar(toolbar)
        activity?.supportActionBar?.setDisplayHomeAsUpEnabled(true)
        activity?.supportActionBar?.setDisplayShowHomeEnabled(true)
        activity?.supportActionBar?.setTitle(R.string.acq_cardpay_title)
        if (state.showArrow) {
            toolbar?.setNavigationIcon(R.drawable.acq_arrow_back)
        }
        toolbar?.setNavigationOnClickListener {
            viewModel.handleEvent(PaymentByNewCardViewModel.Event.BackPressed)
        }
    }

    private suspend fun collectState() {
        viewModel.stateFlow.collectLatest {
            val paymentOptions = it.paymentOptions

            initToolbar(it)

            initAppBaseChallengeDelegate(
                requireActivity(), AcquiringSdk(
                    paymentOptions.terminalKey,
                    paymentOptions.publicKey
                ),
                viewLifecycleOwner.lifecycleScope
            )

            val binding = viewBinding ?: return@collectLatest

            val newCardDataInput = binding.newCardDataInput
            newCardDataInput.listener = null
            cardScannerWrapper?.cameraCardScannerContract = paymentOptions.features.cameraCardScannerContract
            newCardDataInput.isScanEnabled = cardScannerWrapper?.isEnabled == true
            newCardDataInput.useSecuredKeyboard = paymentOptions.features.useSecureKeyboard
            newCardDataInput.isVisible = true
            newCardDataInput.showCvcError(it.isValidCvc == false)
            newCardDataInput.showCardNumberError(it.isValidCardNumber == false)
            newCardDataInput.expireDateError(it.isValidExpireDate == false)
            newCardDataInput.listener = object: CardDataInputView.Listener {
                override fun onCardNumberChanged(cardNumber: String) {
                    viewModel.handleEvent(PaymentByNewCardViewModel.Event.CardNumberChanged(cardNumber))
                }

                override fun onExpireDateChanged(expireDate: String) {
                    viewModel.handleEvent(PaymentByNewCardViewModel.Event.ExpireDateChanged(expireDate))
                }

                override fun onCvcChanged(cvc: String) {
                    viewModel.handleEvent(PaymentByNewCardViewModel.Event.CvcChanged(cvc))
                }

                override fun onScanButtonClick() {
                    cardScannerWrapper?.start()
                }

                override fun onFocusChanged(
                    isCardNumberFocused: Boolean,
                    isExpireDateFocused: Boolean,
                    isCvcFocused: Boolean
                ) {
                    viewModel.handleEvent(PaymentByNewCardViewModel.Event.CardFocusChanged(
                        isCardNumberFocused = isCardNumberFocused,
                        isExpireDateFocused = isExpireDateFocused,
                        isCvcFocused= isCvcFocused,
                    ))
                }
            }

            binding.acqPayBtn.apply {
                text = getString(R.string.acq_cardpay_pay, it.amount)
                isEnabled = it.payButtonEnabled
                setOnClickListener {
                    viewModel.handleEvent(PaymentByNewCardViewModel.Event.PayButtonClick)
                }
            }

            handleNotification(it.notification)

            val acqSendReceiptSwitch = binding.acqSendReceiptSwitch
            acqSendReceiptSwitch.isChecked = it.sendReceipt
            acqSendReceiptSwitch.setOnCheckedChangeListener { view, isChecked ->
                if (view.isPressed) {
                    viewModel.handleEvent(PaymentByNewCardViewModel.Event.SendReceiptChanged(isChecked))
                }
            }
            val emailInput = binding.emailInput
            emailInput.setError(it.isValidEmail == false)
            emailInput.isVisible = it.sendReceipt
            emailInput.listener = object : EmailInputView.Listener {
                override fun onEmailChanged(email: String) {
                    viewModel.handleEvent(
                        PaymentByNewCardViewModel.Event.EmailChanged(email)
                    )
                }

                override fun onFocusChanged(isEmailFocused: Boolean) {
                    viewModel.handleEvent(
                        PaymentByNewCardViewModel.Event.EmailFocusChanged(isEmailFocused)
                    )
                }
            }

            handleLoadingInProcess(it.progress)
        }
    }

    private fun onCardScannerResult(result: ScannedCardResult) {
        when(result) {
            ScannedCardResult.Cancel,
            is ScannedCardResult.Failure -> Unit
            is ScannedCardResult.Success -> {
                viewModel.handleEvent(PaymentByNewCardViewModel.Event.CardScanned(
                    cardNumber = result.data.cardNumber,
                    expireDate = result.data.expireDate
                ))
            }
        }
    }

    private fun handleNotification(notification: Notification?) {
        if (notification != null) {
            notificationBottomSheetBinding?.show(
                notification.toStatusViewData(requireContext())
            )
        } else {
            notificationBottomSheetBinding?.hide()
        }
    }

    private fun handleLoadingInProcess(inProcess: Boolean) {
        val binding = viewBinding ?: return
        binding.acqPayBtn.apply {
            isLoading = inProcess
            isClickable = !inProcess
        }
        binding.acqSendReceiptSwitch.apply {
            setDisabledFieldAlpha(inProcess)
            isEnabled = !inProcess
        }
        binding.emailInput.apply {
            setDisabledFieldAlpha(inProcess)
            isEnabled = !inProcess
        }
        binding.newCardDataInput.apply {
            setDisabledFieldAlpha(inProcess)
            isEnabled = !inProcess
        }
    }

    private fun returnResult(result: Result) {
        setFragmentResult(
            RESULT_KEY, bundleOf(
                RESULT_DATA_KEY to result
            )
        )
    }

    companion object {
        private const val RESULT_KEY = "PaymentByNewCardFragment.RESULT_KEY"
        private const val RESULT_DATA_KEY = "PaymentByNewCardFragment.RESULT_DATA_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(RESULT_KEY, lifecycleOwner) { _, bundle ->
                listener.invoke(bundle.getParcelable(RESULT_DATA_KEY)!!)
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data class PaymentSuccess(
            var paymentId: Long,
            var cardId: String? = null,
            var rebillId: String? = null
        ) : Result

        @Parcelize
        data class PaymentError(
            val throwable: Throwable
        ) : Result

        @Parcelize
        data class NeedThreeDs(
            val paymentOptions: PaymentOptions,
            val data: ThreeDsData,
            val paymentSource: PaymentSource,
        ) : Result

        @Parcelize
        data object PaymentCancel : Result
    }
}

