package ru.tinkoff.acquiring.sdk.redesign.payment.ui.v2

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.paysources.CardData
import ru.tinkoff.acquiring.sdk.models.paysources.CardSource
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardProcess
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardState
import ru.tinkoff.acquiring.sdk.redesign.common.emailinput.models.EmailValidator
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBasedTransaction
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeResult
import ru.tinkoff.acquiring.sdk.ui.delegate.CardDataInputDelegate
import ru.tinkoff.acquiring.sdk.ui.fragments.v2.AttachCardViewModelV2.Command
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory

internal class PaymentByNewCardViewModel(
    savedStateHandle: SavedStateHandle,
    private val paymentByCardProcess: PaymentByCardProcess,
    private val cardDataInputDelegate: CardDataInputDelegate,
) : AbstractPaymentByCardViewModel<PaymentByNewCardViewModel.State, PaymentByNewCardViewModel.Command>() {

    private val args: PaymentByNewCardFragmentArgs =
        PaymentByNewCardFragmentArgs.fromSavedStateHandle(savedStateHandle)
    private val paymentOptions = args.paymentOptions
    private val _commandFlow = MutableSharedFlow<Command>()
    override val commandFlow = _commandFlow.asSharedFlow()
    private val _stateFlow: MutableStateFlow<State> = MutableStateFlow(createInitialState())
    override val stateFlow: StateFlow<State> = _stateFlow.asStateFlow()

    fun processChallengeResult(result: AppBaseChallengeResult) = viewModelScope.launch {
        when (result) {
            AppBaseChallengeResult.Cancelled -> {
                _commandFlow.emit(Command.ReturnPaymentCancel)
            }

            is AppBaseChallengeResult.ProtocolError -> {
                onChallengeError(result.error)
            }

            is AppBaseChallengeResult.RuntimeError -> {
                onChallengeError(result.error)
            }

            is AppBaseChallengeResult.Error -> {
                onChallengeError(result.error)
            }

            is AppBaseChallengeResult.Success -> {
                returnSuccessResultOrShowNotification(
                    paymentId = result.paymentResult.paymentId,
                    cardId = result.paymentResult.cardId,
                    rebillId = result.paymentResult.rebillId
                )
            }

            is AppBaseChallengeResult.TimeOut -> {
                val onClick: () -> Unit = {
                    viewModelScope.launch {
                        _stateFlow.update {
                            it.copy(
                                notification = null
                            )
                        }
                        _commandFlow.emit(Command.ReturnPaymentError(result.error))
                    }
                }
                val notification = NotificationFactory.getPaymentTimeoutErrorNotification(
                    exception = result.error,
                    onClickFactory = { onClick }
                )
                _stateFlow.update {
                    it.copy(
                        notification = notification
                    )
                }
            }

            AppBaseChallengeResult.Loading -> {
                val notification = Notification(
                    type = StatusViewData.Type.PROGRESS,
                    title = R.string.acq_commonsheet_processing_title,
                    description = R.string.acq_commonsheet_processing_description,
                )
                _stateFlow.update {
                    it.copy(
                        notification = notification
                    )
                }
            }
        }
    }

    fun handleEvent(event: Event) {
        when (event) {
            Event.BackPressed -> onBackPressed()
            Event.PayButtonClick -> onPayButtonClick()
            is Event.SendReceiptChanged -> sendReceiptChanged(event.isChecked)
            Event.Resume -> onViewResume()
            is Event.EmailChanged -> onEmailChanged(event.email)
            is Event.CvcChanged -> onCvcChanged(event.cvc)
            is Event.ExpireDateChanged -> onCardExpireDateChanged(event.expireDate)
            is Event.CardNumberChanged -> onCardNumberChanged(event.cardNumber)
            is Event.CardScanned -> onCardScanned(event.cardNumber, event.expireDate)
            is Event.CardFocusChanged -> onCardFocusChanged(event.isCardNumberFocused, event.isExpireDateFocused, event.isCvcFocused)
            is Event.EmailFocusChanged -> onEmailFocusChanged(event.isEmailFocused)
        }
    }

    private fun onEmailFocusChanged(emailFocused: Boolean) {
        val currentState = stateFlow.value
        _stateFlow.update {
            it.copy(isEmailFocused = emailFocused)
        }
        if (currentState.isEmailFocused != emailFocused) {
            validateEmail(
                email = currentState.email.orEmpty(),
                forceHideError = emailFocused
            )
        }
    }

    private fun onCardFocusChanged(
        cardNumberFocused: Boolean,
        expireDateFocused: Boolean,
        cvcFocused: Boolean
    ) {
        val currentState = stateFlow.value
        _stateFlow.update { it.copy(
            isCardNumberFocused = cardNumberFocused,
            isExpireDateFocused = expireDateFocused,
            isCvcFocused = cvcFocused
        ) }
        if (currentState.isCardNumberFocused != cardNumberFocused) {
            validateCardNumberAndGoToNextField(
                cardNumber = currentState.cardNumber.orEmpty(),
                gotoNextField = false,
                forceHideError = cardNumberFocused,
                isFocused = cardNumberFocused
            )
        }
        if (currentState.isExpireDateFocused != expireDateFocused) {
            validateExpireDateAndFocusNextField(
                expireDate = currentState.expireDate.orEmpty(),
                allowGoToNextField = false
            )
        }
        if (currentState.isCvcFocused != cvcFocused) {
            validateCvc(currentState.cvc.orEmpty())
        }
    }

    override fun onViewResume(isFirstTime: Boolean) {
        if (isFirstTime) {
            viewModelScope.launch {
                val email = _stateFlow.value.email
                if (!email.isNullOrBlank()) {
                    _commandFlow.emit(
                        Command.FillEmailField(
                            email = email,
                        )
                    )
                }
                _commandFlow.emit(Command.RequestCardNumberFieldFocus)
            }
        } else {
            viewModelScope.launch {
                val currentState = _stateFlow.value
                val email = currentState.email
                if (!email.isNullOrBlank()) {
                    _commandFlow.emit(
                        Command.FillEmailField(
                            email = email,
                        )
                    )
                }
                _commandFlow.emit(
                    Command.FillCardData(
                        cardNumber = currentState.cardNumber,
                        expireDate = currentState.expireDate,
                        cvc = currentState.cvc,
                    )
                )
            }
        }
    }

    private fun createInitialState(): State {
        val email = paymentOptions.customer.email
        return State(
            showArrow = args.showArrow,
            isValidEmail = email?.let { EmailValidator.validate(it) } ?: true,
            sendReceipt = email.isNullOrBlank().not(),
            email = email,
            paymentOptions = paymentOptions,
        )
    }

    private fun sendReceiptChanged(isSelect: Boolean) {
        _stateFlow.update {
            it.copy(
                sendReceipt = isSelect,
                isValidEmail = EmailValidator.validate(it.email)
            )
        }

        viewModelScope.launch {
            _commandFlow.emit(Command.ClearFocus)
        }
    }

    private fun onEmailChanged(email: String) {
        validateEmail(
            email = email,
            forceHideError = false
        )
    }

    private fun validateEmail(email: String, forceHideError: Boolean) {
        var isValid: Boolean? = EmailValidator.validate(email)
        if (isValid == false && (!shouldDisplayEmailError(email) || forceHideError)) {
            isValid = null
        }
        _stateFlow.update {
            it.copy(
                email = email,
                isValidEmail = isValid
            )
        }
    }

    private fun shouldDisplayEmailError(email: String): Boolean {
        return !stateFlow.value.isEmailFocused
    }

    private fun onChallengeError(error: Throwable) {
        val onClick: () -> Unit = {
            viewModelScope.launch {
                _commandFlow.emit(Command.ReturnPaymentError(error))
            }
        }

        _stateFlow.update {
            it.copy(
                notification = NotificationFactory.getPaymentErrorNotification(
                    exception = error,
                    onClickFactory = { onClick },
                    options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
                )
            )
        }
    }

    private fun pay() {
        val currentState = _stateFlow.value
        val cardData = currentState.cardSource
        if (cardData == null) {
            viewModelScope.launch {
                finishWithPaymentSourceError()
            }
            return
        }

        viewModelScope.launch {
            paymentByCardProcess.state.collectLatest {
                when (it) {
                    is PaymentByCardState.Started -> setProgress(true)
                    is PaymentByCardState.Created -> Unit
                    is PaymentByCardState.Error -> returnErrorOrShowNotification(it.throwable)
                    is PaymentByCardState.Success -> {
                        returnSuccessResultOrShowNotification(
                            paymentId = it.paymentId,
                            cardId = it.cardId,
                            rebillId = it.rebillId
                        )
                    }

                    is PaymentByCardState.ThreeDsUiNeeded -> {
                        viewModelScope.launch {
                            val paymentSource = _stateFlow.value.cardSource
                            if (paymentSource == null) {
                                finishWithPaymentSourceError()
                            } else {
                                _commandFlow.emit(
                                    Command.RequestThreeDs(
                                        paymentOptions = paymentOptions,
                                        data = it.threeDsState.data,
                                        paymentSource = paymentSource
                                    )
                                )
                            }
                        }
                        goTo3ds()
                    }

                    is PaymentByCardState.ThreeDsInProcess -> setProgress(false)

                    is PaymentByCardState.ThreeDsAppBase -> {
                        viewModelScope.launch {
                            setProgress(true)
                            _commandFlow.emit(
                                Command.RequestThreeAppBaseChallenge(
                                    transaction = it.transaction,
                                    threeDsData = it.threeDsData,
                                )
                            )
                        }
                    }

                    else -> Unit
                }
            }
        }

        val emailForPayment = if (currentState.sendReceipt) currentState.email else null
        paymentByCardProcess.start(
            cardData = cardData,
            paymentOptions = currentState.paymentOptions,
            email = emailForPayment
        )
    }

    private suspend fun finishWithPaymentSourceError() {
        val throwable = IllegalStateException("paymentSource is null")
        AcquiringSdk.log(throwable)
        _commandFlow.emit(Command.ReturnPaymentError(throwable))
    }

    private fun returnErrorOrShowNotification(error: Throwable) {
        val onClick = {
            _stateFlow.update {
                it.copy(notification = null)
            }
            returnErrorResult(error)
        }
        if (paymentOptions.features.showPaymentNotifications) {
            val notification = NotificationFactory.getPaymentErrorNotification(
                exception = error,
                onClickFactory = { onClick },
                options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
            )
            _stateFlow.update {
                it.copy(
                    notification = notification
                )
            }
        } else {
            onClick()
        }
    }

    private fun returnSuccessResultOrShowNotification(
        paymentId: Long,
        cardId: String?,
        rebillId: String?
    ) {
        val onCLick = {
            returnSuccessResult(
                paymentId = paymentId,
                cardId = cardId,
                rebillId = rebillId
            )
        }
        if (paymentOptions.features.showPaymentNotifications) {
            val notification =
                NotificationFactory.getPaymentSuccessCommonNotification(
                    amount = paymentOptions.order.amount,
                    onClick = onCLick
                )
            _stateFlow.update {
                it.copy(
                    notification = notification
                )
            }
        } else {
            onCLick()
        }
    }

    private fun returnSuccessResult(paymentId: Long, cardId: String?, rebillId: String?) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnPaymentSuccess(
                    paymentId = paymentId,
                    cardId = cardId,
                    rebillId = rebillId
                )
            )
        }
    }

    private fun returnErrorResult(throwable: Throwable) {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnPaymentError(throwable))
        }
    }

    private fun setProgress(progress: Boolean) {
        _stateFlow.update {
            it.copy(progress = progress)
        }
    }

    private fun goTo3ds() {
        paymentByCardProcess.goTo3ds()
    }

    private fun cancelPayment() {
        paymentByCardProcess.stop()
    }

    private fun onCardScanned(cardNumber: String?, expireDate: String?) {
        cardDataInputDelegate.onCardScanned(
            cardNumber = cardNumber,
            expireDate = expireDate,
            updateState = { cardNumber, expireDate ->
                viewModelScope.launch {
                    _commandFlow.emit(
                        Command.FillCardData(
                            cardNumber = cardNumber,
                            expireDate = expireDate
                        )
                    )
                }
            },
            requestCvcFocus = {
                viewModelScope.launch {
                    _commandFlow.emit(Command.RequestCvcFieldFocus)
                }
            },
            requestExpireDateFocus = {
                viewModelScope.launch {
                    _commandFlow.emit(Command.RequestExpireDateFieldFocus)
                }
            }
        )
    }

    private fun onCardNumberChanged(cardNumber: String) {
        validateCardNumberAndGoToNextField(
            cardNumber = cardNumber,
            gotoNextField = true,
            forceHideError = false,
            isFocused = stateFlow.value.isCardNumberFocused,
        )
    }

    private fun validateCardNumberAndGoToNextField(
        cardNumber: String,
        gotoNextField: Boolean,
        forceHideError: Boolean,
        isFocused: Boolean,
    ) {
        cardDataInputDelegate.validateCardNumberAndGoToNextField(
            cardNumber = cardNumber,
            allowGotoNextField = gotoNextField,
            forceHideError = forceHideError,
            isFocused = isFocused,
            updateState = { normalizedCardNumber, isValidCardNumber ->
                _stateFlow.update {
                    it.copy(
                        cardNumber = normalizedCardNumber,
                        isValidCardNumber = isValidCardNumber
                    )
                }
            },
            goToNextField = {
                viewModelScope.launch {
                    _commandFlow.emit(Command.RequestExpireDateFieldFocus)
                }
            }
        )
    }

    override fun validateCvc(cvc: String) {
        val isFocused = stateFlow.value.isCvcFocused
        cardDataInputDelegate.validateCvc(
            cvc = cvc,
            isFocused = isFocused,
            updateState = { cvc, isCvcValid ->
                _stateFlow.update {
                    it.copy(
                        cvc = cvc,
                        isValidCvc = isCvcValid
                    )
                }
            }
        )
    }

    private fun onCardExpireDateChanged(expireDate: String) {
        validateExpireDateAndFocusNextField(expireDate, true)
    }

    private fun validateExpireDateAndFocusNextField(expireDate: String, allowGoToNextField: Boolean ) {
        val isFocused = stateFlow.value.isExpireDateFocused
        cardDataInputDelegate.validateExpireDateAndFocusNextField(
            expireDate = expireDate,
            isFocused = isFocused,
            allowGoToNextField = allowGoToNextField,
            goToNextField = ::gotoCvcFieldOrClearFocus,
            updateState = { expireDate, isValidExpireDate ->
                _stateFlow.update {
                    it.copy(
                        expireDate = expireDate,
                        isValidExpireDate = isValidExpireDate
                    )
                }
            }
        )
    }

    private fun gotoCvcFieldOrClearFocus() {
        val cvc = _stateFlow.value.cvc
        if (cvc.isNullOrBlank()) {
            viewModelScope.launch {
                _commandFlow.emit(Command.RequestCvcFieldFocus)
            }
        } else {
            viewModelScope.launch {
                _commandFlow.emit(Command.ClearFocus)
            }
        }
    }

    private fun onPayButtonClick() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ClearFocus)
            pay()
        }
    }

    private fun onBackPressed() {
        cancelPayment()
        returnPaymentCancel()
    }

    private fun returnPaymentCancel() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnPaymentCancel)
        }
    }


    data class State(
        val showArrow: Boolean = false,
        val paymentOptions: PaymentOptions,
        val progress: Boolean = false,
        val notification: Notification? = null,
        val cardNumber: String? = null,
        val isCardNumberFocused: Boolean = false,
        val isValidCardNumber: Boolean? = null,
        val expireDate: String? = null,
        val isExpireDateFocused: Boolean = false,
        val isValidExpireDate: Boolean? = null,
        val cvc: String? = null,
        val isCvcFocused: Boolean = false,
        val isValidCvc: Boolean? = null,
        val sendReceipt: Boolean = false,
        val email: String? = null,
        val isValidEmail: Boolean? = null,
        val isEmailFocused: Boolean = false,
    ) {
        val payButtonEnabled: Boolean
            get() {
                return if (sendReceipt) {
                    isValidEmail == true && isValidCvc == true && isValidCardNumber == true && isValidExpireDate == true
                } else {
                    isValidCvc == true && isValidCardNumber == true && isValidExpireDate == true
                }
            }

        val amount = paymentOptions.order.amount.toHumanReadableString()

        val cardSource: CardSource?
            get() {
                return if (cardNumber != null && expireDate != null && cvc != null) {
                    CardData(cardNumber, expireDate, cvc)
                } else {
                    null
                }
            }
    }

    sealed interface Event {
        data object Resume : Event
        data object BackPressed : Event
        data object PayButtonClick : Event
        class CardNumberChanged(
            val cardNumber: String
        ) : Event
        class ExpireDateChanged(
            val expireDate: String
        ) : Event
        class CvcChanged(
            val cvc: String
        ) : Event
        class SendReceiptChanged(
            val isChecked: Boolean,
        ) : Event
        class EmailChanged(
            val email: String
        ) : Event
        class CardScanned(
            val cardNumber: String,
            val expireDate: String,
        ) : Event
        class CardFocusChanged(
            val isCardNumberFocused: Boolean,
            val isExpireDateFocused: Boolean,
            val isCvcFocused: Boolean
        ) : Event

        class EmailFocusChanged(
            val isEmailFocused: Boolean
        ) : Event
    }

    sealed interface Command {
        class ReturnPaymentError(
            val throwable: Throwable
        ) : Command

        class RequestThreeDs(
            val paymentOptions: PaymentOptions,
            val data: ThreeDsData,
            val paymentSource: PaymentSource
        ) : Command

        class RequestThreeAppBaseChallenge(
            val transaction: ThreeDsAppBasedTransaction,
            val threeDsData: ThreeDsData,
        ) : Command

        class ReturnPaymentSuccess(
            val paymentId: Long,
            val cardId: String?,
            val rebillId: String?
        ) : Command

        class FillCardData(
            val cardNumber: String?,
            val expireDate: String?,
            val cvc: String? = null,
        ) : Command

        class FillEmailField(
            val email: String?
        ) : Command

        data object ReturnPaymentCancel : Command
        data object RequestEmailFieldFocus : Command
        data object RequestCardNumberFieldFocus : Command
        data object RequestExpireDateFieldFocus : Command
        data object RequestCvcFieldFocus : Command
        data object ClearFocus : Command
    }
}
