package ru.tinkoff.acquiring.sdk.redesign.payment.ui.v2

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.addCallback
import androidx.appcompat.app.AppCompatActivity
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.setFragmentResult
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentPaymentByChosenCardBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.keyboard.SecureKeyboardController
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegate
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegateImpl
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.utils.KeyboardVisionUtils
import ru.tinkoff.acquiring.sdk.utils.setDisabledFieldAlpha
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

/**
 * Created by v.budnitskiy
 */
internal class PaymentBySavedCardFragment : Fragment(),
    AppBaseChallengeDelegate by AppBaseChallengeDelegateImpl(), IsolatedKoinComponent {

    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? =
        null
    private var viewBinding: AcqFragmentPaymentByChosenCardBinding? = null

    private val viewModel by viewModel<PaymentBySavedCardViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return AcqFragmentPaymentByChosenCardBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.handleEvent(PaymentBySavedCardViewModel.Event.BackPressed)
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                collectState()
            }
        }

        this.notificationBottomSheetBinding = NotificationUtils.bindNotificationBottomSheet(
            activity = requireActivity()
        )

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.CREATED) {
                collectCommand()
            }
        }
    }

    override fun onResume() {
        super.onResume()

        viewBinding?.let {
            SecureKeyboardController.getInstance().apply {
                setContentContainer(it.acqPayByCardContentWrapper)
                registerInsets(it.root)
            }
        }
        viewModel.handleEvent(PaymentBySavedCardViewModel.Event.Resume)
    }

    override fun onPause() {
        super.onPause()
        SecureKeyboardController.getInstance().clear()
    }

    override fun onDestroy() {
        super.onDestroy()
        (requireActivity() as? AppCompatActivity)?.setSupportActionBar(null)
    }

    override fun onDestroyView() {
        viewBinding = null
        super.onDestroyView()
    }

    private suspend fun collectCommand(): Nothing {
        viewModel.commandFlow.collect {
            when (it) {
                is PaymentBySavedCardViewModel.Command.ChangeCard -> {
                    returnResult(
                        Result.NeedChangeCard(
                            savedCardsOptions = it.savedCardsOptions,
                            paymentOptions = it.paymentOptions
                        )
                    )
                }

                is PaymentBySavedCardViewModel.Command.RequestThreeDs -> {
                    returnResult(
                        Result.NeedThreeDs(
                            paymentOptions = it.paymentOptions,
                            data = it.data,
                            paymentSource = it.paymentSource
                        )
                    )
                }

                is PaymentBySavedCardViewModel.Command.RequestThreeAppBaseChallenge -> {
                    launchChallenge(
                        transaction = it.transaction,
                        threeDsData = it.threeDsData,
                        onResult = viewModel::processChallengeResult
                    )
                }

                PaymentBySavedCardViewModel.Command.ReturnPaymentCancel -> {
                    finishWithCancel()
                }

                is PaymentBySavedCardViewModel.Command.ReturnPaymentError -> {
                    finishWithError(it.throwable)
                }

                is PaymentBySavedCardViewModel.Command.ReturnPaymentSuccess -> {
                    finishWithSuccess(
                        paymentId = it.paymentId,
                        cardId = it.cardId,
                        rebillId = it.rebillId
                    )
                }

                PaymentBySavedCardViewModel.Command.RequestEmailFieldFocus -> {
                    viewBinding?.emailInput?.requestFocus()
                }

                PaymentBySavedCardViewModel.Command.RequestCvcFieldFocus -> {
                    setFocusToCvc()
                }

                PaymentBySavedCardViewModel.Command.ClearFocus -> {
                    KeyboardVisionUtils.hideKeyboard(requireView())
                    viewBinding?.emailInput?.clearFocus()
                    viewBinding?.acqChosenCard?.clearFocus()
                }

                is PaymentBySavedCardViewModel.Command.FillEmailField -> {
                    viewBinding?.emailInput?.setData(it.email)
                }
            }
        }
    }

    private fun setFocusToCvc() {
        viewBinding?.acqChosenCard?.focusCvc()
    }

    private fun onChangeCard() {
        viewModel.handleEvent(PaymentBySavedCardViewModel.Event.ChangeCard)
    }

    private fun finishWithCancel() {
        returnResult(Result.PaymentCancel)
    }

    private fun finishWithError(throwable: Throwable) {
        returnResult(Result.PaymentError(throwable))
    }

    private fun finishWithSuccess(paymentId: Long, cardId: String?, rebillId: String?) {
        returnResult(
            Result.PaymentSuccess(
                paymentId = paymentId,
                cardId = cardId,
                rebillId = rebillId,
            )
        )
    }

    private fun initToolbar(state: PaymentBySavedCardViewModel.State) {
        val activity = (requireActivity() as? AppCompatActivity) ?: return
        val toolbar = viewBinding?.acqToolbar ?: return
        activity.setSupportActionBar(toolbar)
        activity.supportActionBar?.apply {
            setDisplayHomeAsUpEnabled(true)
            setDisplayShowHomeEnabled(true)
            setTitle(R.string.acq_cardpay_title)
        }
        if (state.showArrow) {
            toolbar.setNavigationIcon(R.drawable.acq_arrow_back)
        }
        toolbar.setNavigationOnClickListener {
            viewModel.handleEvent(PaymentBySavedCardViewModel.Event.BackPressed)
        }
    }

    private suspend fun collectState() {
        viewModel.stateFlow.collectLatest {
            val paymentOptions = it.paymentOptions

            initToolbar(it)

            initAppBaseChallengeDelegate(
                requireActivity(), AcquiringSdk(
                    paymentOptions.terminalKey,
                    paymentOptions.publicKey
                ),
                viewLifecycleOwner.lifecycleScope
            )

            val binding = viewBinding ?: return@collectLatest
            binding.acqPayBtn.apply {
                text = getString(R.string.acq_cardpay_pay, it.amount)
                isEnabled = it.payButtonEnabled
                setOnClickListener {
                    viewModel.handleEvent(PaymentBySavedCardViewModel.Event.PayButtonClick)
                }
            }

            handleNotification(it.notification)

            val acqChosenCard = binding.acqChosenCard
            acqChosenCard.isVisible = true
            acqChosenCard.setData(
                number = it.chosenCard.pan!!,
                bankName = it.chosenCard.bankName
            )
            acqChosenCard.setCvcError(it.isValidCvc == false)
            acqChosenCard.listener = object : ChosenCardView.Listener{
                override fun onChangeCard() {
                    this@PaymentBySavedCardFragment.onChangeCard()
                }

                override fun onCvcChanged(cvc: String) {
                    viewModel.handleEvent(
                        PaymentBySavedCardViewModel.Event.ChosenCardCvcChanged(cvc = cvc)
                    )
                }

                override fun onFocusChanged(isCvcFocused: Boolean) {
                    viewModel.handleEvent(
                        PaymentBySavedCardViewModel.Event.CvcFocusChanged(isCvcFocused)
                    )
                }
            }

            val acqSendReceiptSwitch = binding.acqSendReceiptSwitch
            acqSendReceiptSwitch.isChecked = it.sendReceipt
            acqSendReceiptSwitch.setOnCheckedChangeListener { view, isChecked ->
                if (view.isPressed) {
                    viewModel.handleEvent(PaymentBySavedCardViewModel.Event.SendReceiptChanged(isChecked))
                }
            }
            val emailInput = binding.emailInput
            emailInput.setError(it.isValidEmail == false)
            emailInput.isVisible = it.sendReceipt
            emailInput.listener = object: EmailInputView.Listener{
                override fun onEmailChanged(email: String) {
                    viewModel.handleEvent(
                        PaymentBySavedCardViewModel.Event.EmailChanged(email)
                    )
                }

                override fun onFocusChanged(isEmailFocused: Boolean) {
                    viewModel.handleEvent(
                        PaymentBySavedCardViewModel.Event.EmailFocusChanged(isEmailFocused)
                    )
                }
            }

            handleLoadingInProcess(it.progress)
        }
    }

    private fun handleNotification(notification: Notification?) {
        if (notification != null) {
            notificationBottomSheetBinding?.show(
                notification.toStatusViewData(requireContext())
            )
        } else {
            notificationBottomSheetBinding?.hide()
        }
    }

    private fun handleLoadingInProcess(inProcess: Boolean) {
        val binding = viewBinding ?: return
        binding.acqPayBtn.apply {
            isLoading = inProcess
            isClickable = !inProcess
        }

        binding.acqSendReceiptSwitch.apply {
            isEnabled = !inProcess
            setDisabledFieldAlpha(inProcess)
        }
        binding.emailInput.apply {
            isEnabled = !inProcess
            setDisabledFieldAlpha(inProcess)
        }
        binding.acqChosenCard.apply {
            isEnabled = !inProcess
            setDisabledFieldAlpha(inProcess)
        }
    }

    private fun returnResult(result: Result) {
        setFragmentResult(
            RESULT_KEY, bundleOf(
                RESULT_DATA_KEY to result
            )
        )
    }

    companion object {
        private const val RESULT_KEY = "PaymentBySavedCardFragment.RESULT_KEY"
        private const val RESULT_DATA_KEY = "PaymentBySavedCardFragment.RESULT_DATA_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(RESULT_KEY, lifecycleOwner) { _, bundle ->
                listener.invoke(bundle.getParcelable(RESULT_DATA_KEY)!!)
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data class PaymentSuccess(
            var paymentId: Long,
            var cardId: String? = null,
            var rebillId: String? = null
        ) : Result

        @Parcelize
        data class PaymentError(
            val throwable: Throwable
        ) : Result

        @Parcelize
        data class NeedThreeDs(
            val paymentOptions: PaymentOptions,
            val data: ThreeDsData,
            val paymentSource: PaymentSource,
        ) : Result

        @Parcelize
        data class NeedChangeCard(
            val savedCardsOptions: SavedCardsOptions,
            val paymentOptions: PaymentOptions
        ) : Result

        @Parcelize
        data object PaymentCancel : Result
    }
}

